import api from '../../../api';
import {changePagePopupType, changePagePopupVisibility, domikIsLoading, changeCaptchaState} from '../';
import {
    changeCheckPhoneStatus,
    setErrors,
    clearErrors,
    setSuggestedAccounts,
    switchConfirmationMethod,
    changePhoneOverheatStatus,
    updateCallingNumberTemplate,
    changePhoneValidationStatus
} from './';
import switchToModeRestoreLoginResult from '../switchToModeRestoreLoginResult';
import {handleCheckPhoneErrors} from './handleErrors';
import metrics from '../../../metrics';
import {
    RESTORE_LOGIN_ENTER_PHONE_SUCCESS,
    RESTORE_LOGIN_ENTER_PHONE_FAIL,
    RESTORE_LOGIN_ENTER_CAPTCHA_FAIL,
    RESTORE_LOGIN_PAGE,
    RESTORE_LOGIN_CONFIRM_PHONE_WITH_CALL,
    RESTORE_LOGIN_CONFIRM_PHONE_WITH_FLASH_CALL,
    RESTORE_LOGIN_CONFIRM_PHONE_WITH_SMS,
    RESTORE_LOGIN_FAIL_SMS,
    RESTORE_LOGIN_FAIL_CALL,
    RESTORE_LOGIN_SUCCESS_SMS,
    RESTORE_LOGIN_SUCCESS_SMS_RETRY,
    RESTORE_LOGIN_SUCCESS_CALL,
    RESTORE_LOGIN_SUCCESS_CALL_RETRY
} from '../../metrics_constants';

export const checkPhone = {
    check(isCodeResend = false) {
        return (dispatch, getState) => {
            const {
                restoreLogin: {
                    phone,
                    confirmMethod,
                    isValidPhone,
                    isPhoneOverheated,
                    errors: {code: codeError, phone: phoneError}
                },
                common: {track_id: trackId},
                settings: {language}
            } = getState();
            const params = {
                phone_number: phone,
                track_id: trackId,
                display_language: language,
                confirm_method: confirmMethod
            };
            const isCallsLimitExceededError = codeError === 'callsLimitExceeded' || phoneError === 'callsLimitExceeded';

            let isSMSConfirmation = confirmMethod === 'by_sms';

            if (!isSMSConfirmation && (isCallsLimitExceededError || isPhoneOverheated)) {
                dispatch(switchConfirmationMethod('by_sms'));
                params.confirm_method = 'by_sms';
                isSMSConfirmation = true;
            }

            dispatch(clearErrors());
            dispatch(domikIsLoading(true));

            if (!isSMSConfirmation && !isValidPhone) {
                dispatch(this.validate(isCodeResend));
                dispatch(domikIsLoading(false));
                return;
            }

            api.request('auth/restore_login/check_phone', params)
                .done((response = {}) => {
                    dispatch(changeCheckPhoneStatus(true));
                    dispatch(changePagePopupType('restoreLoginEnterConfirmationCode'));
                    dispatch(changePagePopupVisibility(true));
                    metrics.send([RESTORE_LOGIN_PAGE, RESTORE_LOGIN_ENTER_PHONE_SUCCESS]);

                    if (response.calling_number_template) {
                        dispatch(updateCallingNumberTemplate(response.calling_number_template));
                    }

                    sendMetrics(getState(), isCodeResend);
                })
                .fail((err) => {
                    const errors = err.errors || [];

                    if (errors.includes('user.not_verified')) {
                        dispatch(changePagePopupType('restoreLoginEnterCaptcha'));
                        dispatch(changePagePopupVisibility(true));
                        dispatch(changeCaptchaState(true));
                        return;
                    }

                    if (errors.includes('phone_secure.not_found')) {
                        dispatch(changePagePopupVisibility(false));
                        dispatch(changePagePopupType(''));
                        dispatch(setSuggestedAccounts([]));
                        dispatch(switchToModeRestoreLoginResult());
                    }

                    if (errors.includes('calls_limit.exceeded')) {
                        dispatch(changePhoneOverheatStatus(true));
                    }

                    dispatch(setErrors(handleCheckPhoneErrors(errors)));

                    const isCaptchaError = errors.some(
                        (apiError) => apiError === 'captcha.not_matched' || apiError === 'captcha.empty'
                    );

                    metrics.send(
                        isCaptchaError
                            ? [RESTORE_LOGIN_PAGE, RESTORE_LOGIN_ENTER_CAPTCHA_FAIL]
                            : [RESTORE_LOGIN_PAGE, RESTORE_LOGIN_ENTER_PHONE_FAIL, errors[0]]
                    );

                    sendMetrics(getState(), isCodeResend, errors);
                })
                .always(() => dispatch(domikIsLoading(false)));
        };
    },

    validate(isCodeResend) {
        return (dispatch, getState) => {
            const {
                restoreLogin: {phone},
                common: {track_id: trackId, csrf}
            } = getState();
            const params = {
                phone_number: phone,
                validate_for_call: true,
                check_speech: true,
                track_id: trackId,
                csrf_token: csrf
            };

            const onFail = () => {
                dispatch(switchConfirmationMethod('by_sms'));
                dispatch(changePhoneValidationStatus(false));
            };

            api.request('auth/validate_phone', params)
                .done((res) => {
                    const isValid = Boolean(res.valid_for_call) && res.status === 'ok';

                    if (!isValid) {
                        onFail();
                        return;
                    }

                    dispatch(changePhoneValidationStatus(true));
                })
                .fail(onFail)
                .always(() => dispatch(this.check(isCodeResend)));
        };
    }
};

export function sendMetrics(state, isCodeResend, errors) {
    const {
        restoreLogin: {isCallExp, isFlashCallExp, confirmMethod}
    } = state;
    const metricsParams = [RESTORE_LOGIN_PAGE];
    const isSMSConfirmation = confirmMethod === 'by_sms';

    if (isCallExp) {
        metricsParams.push(RESTORE_LOGIN_CONFIRM_PHONE_WITH_CALL);
    } else if (isFlashCallExp) {
        metricsParams.push(RESTORE_LOGIN_CONFIRM_PHONE_WITH_FLASH_CALL);
    } else {
        metricsParams.push(RESTORE_LOGIN_CONFIRM_PHONE_WITH_SMS);
    }

    if (Array.isArray(errors) && errors.length) {
        metricsParams.push(isSMSConfirmation ? RESTORE_LOGIN_FAIL_SMS : RESTORE_LOGIN_FAIL_CALL);
        metricsParams.push(errors.join(','));
    } else {
        const callParam = isCodeResend ? RESTORE_LOGIN_SUCCESS_CALL_RETRY : RESTORE_LOGIN_SUCCESS_CALL;
        const smsParam = isCodeResend ? RESTORE_LOGIN_SUCCESS_SMS_RETRY : RESTORE_LOGIN_SUCCESS_SMS;

        metricsParams.push(isSMSConfirmation ? smsParam : callParam);
    }

    metrics.send(metricsParams);
}
