import api from '@blocks/api';
import {getAdvancedError, getError} from '@blocks/utils';
import {registerLiteStateFilter} from '@blocks/filters/registerLiteState';
import {getFormErrors, getFormValues, getLanguage, getRedirectUrl} from '@blocks/selectors';
import {domikIsLoading} from '@blocks/auth/actions';
import {updateErrors, updateValues} from '@blocks/actions/form';
import {FIELDS_NAMES} from '@components/Field';
import metrika from '@blocks/metrics';
import {SOCIAL} from '@blocks/authv2/metrics_constants';
import {amSetAnalyticsToRetpath} from '@blocks/authv2/actions/nativeMobileApi';

export const INIT_SOCIAL_SUGGEST_PROCESS = '[SOCIAL_SUGGEST] INIT_SOCIAL_SUGGEST_PROCESS';
export const SET_REGISTER_LITE_STATE = '[SOCIAL_SUGGEST] SET_REGISTER_LITE_STATE';
export const SET_REGISTER_SOCIAL_STATE = '[SOCIAL_SUGGEST] SET_REGISTER_SOCIAL_STATE';
export const CHANGE_STATE = '[SOCIAL_SUGGEST] CHANGE_STATE';
export const CLEAR_STATE = '[SOCIAL_SUGGEST] CLEAR_STATE';
export const EDIT_FORM_STATE = '[SOCIAL_SUGGEST] EDIT_FORM_STATE';

export const initSocialSuggestProcess = (payload = {}) => ({type: INIT_SOCIAL_SUGGEST_PROCESS, payload});
export const changeState = (payload = {}) => (dispatch) => dispatch({type: CHANGE_STATE, payload});
export const clearState = () => (dispatch) => dispatch({type: CLEAR_STATE});
export const setEditFormState = () => (dispatch) => dispatch({type: EDIT_FORM_STATE});

export const getRegisterLiteState = () => (dispatch, getState) => {
    const state = getState();
    const {socialSuggest = {}} = state;
    const {registerLiteTrackId} = socialSuggest;
    const language = getLanguage(state);

    api.getRegisterLiteState({trackId: registerLiteTrackId, language})
        .then((response = {}) => {
            const data = registerLiteStateFilter(response);
            const {firstname, lastname} = data;

            dispatch(updateValues({field: FIELDS_NAMES.FIRSTNAME, value: firstname}));
            dispatch(updateValues({field: FIELDS_NAMES.LASTNAME, value: lastname}));
            dispatch({type: SET_REGISTER_LITE_STATE, payload: data});
        })
        .catch((errorResponse = {}) => {
            const {error} = errorResponse;

            dispatch({type: SET_REGISTER_LITE_STATE, payload: getAdvancedError('registration', error[0])});
        });
};

export const registerLite = () => (dispatch, getState) => {
    const fields = ['firstname', 'lastname'];
    const state = getState();
    const {socialSuggest = {}, am = {}} = state;
    const {registerLiteState = {}, registerLiteTrackId} = socialSuggest;
    const {email} = registerLiteState;

    if (am.isAm) {
        dispatch(amSetAnalyticsToRetpath('magic_link_reg'));
    }

    const redirectUrl = getRedirectUrl(state);

    const values = getFormValues(state);
    const errors = getFormErrors(state);
    const {firstname, lastname} = values;
    const formErrors = [];

    const params = {
        track_id: registerLiteTrackId,
        firstname,
        lastname,
        eula_accepted: 'on',
        login: email,
        retpath: redirectUrl
    };

    fields.forEach((field) => {
        if (!values[field]) {
            formErrors.push({field, code: 'missingvalue'});

            return dispatch(
                updateErrors({
                    field,
                    error: getError(field, 'missingvalue')
                })
            );
        }

        if (errors[field] && errors[field].code) {
            formErrors.push({field, code: errors[field].code});
        }
    });

    if (formErrors.length) {
        return;
    }

    dispatch(domikIsLoading(true));

    api.registerLite(params)
        .then((response = {}) => {
            const {status, errors, retpath} = response;

            if (status === 'ok') {
                metrika.send([SOCIAL, 'Успешное подтверждение регистрации лайта']);
                return location.replace(retpath || redirectUrl);
            }

            throw errors;
        })
        .catch((errorResponse = {}) => {
            const {error = []} = errorResponse;
            const fieldsErrors = [];

            error.forEach((code = '') => {
                fields.forEach((field = '') => {
                    if (code.startsWith(field)) {
                        fieldsErrors.push({
                            field,
                            error: getError(field, code)
                        });
                    }
                });
            });

            if (fieldsErrors.length) {
                fieldsErrors.forEach((error) => dispatch(updateErrors(error)));
                dispatch(setEditFormState());
            } else {
                dispatch({type: SET_REGISTER_LITE_STATE, payload: getAdvancedError('registration', error[0])});
            }

            dispatch(domikIsLoading(false));
            metrika.send([SOCIAL, 'Ошибка регистрации лайта']);
        });
};

export const registerSocial = () => (dispatch, getState) => {
    const state = getState();
    const {socialSuggest = {}, captcha = {}, form = {}} = state;
    const {registerSocialTrackId, profile = {}} = socialSuggest;
    const redirectUrl = getRedirectUrl(state);
    const values = getFormValues(state);
    const errors = getFormErrors(state);
    const {firstname, lastname} = values;
    const {name} = profile;
    const formErrors = [];
    const fields = name ? [] : ['firstname', 'lastname'];

    const params = {
        track_id: registerSocialTrackId,
        eula_accepted: 'on',
        captcha: values.captcha,
        key: captcha.key,
        retpath: redirectUrl
    };

    if (!name) {
        params.firstname = firstname;
        params.lastname = lastname;
    }

    if (form.keepUnsubscribedValue) {
        params.unsubscribe_from_maillists = form.keepUnsubscribedValue;
    }

    fields.forEach((field) => {
        if (!values[field]) {
            formErrors.push({field, code: 'missingvalue'});

            return dispatch(
                updateErrors({
                    field,
                    error: getError(field, 'missingvalue')
                })
            );
        }

        if (errors[field] && errors[field].code) {
            formErrors.push({field, code: errors[field].code});
        }
    });

    if (formErrors.length) {
        return;
    }

    dispatch(domikIsLoading(true));

    api.registerSocial(params)
        .then((response = {}) => {
            const {status, errors, retpath} = response;

            if (status === 'ok') {
                metrika.send([SOCIAL, 'Успешное подтверждение регистрации социальщика']);
                return location.replace(retpath || redirectUrl);
            }

            throw errors;
        })
        .catch((errorResponse = {}) => {
            const {errors = []} = errorResponse;

            dispatch(domikIsLoading(false));

            if (errors[0] && errors[0].code && errors[0].code.startsWith('captcha')) {
                return dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.CAPTCHA,
                        error: getError(FIELDS_NAMES.CAPTCHA, errors[0].code)
                    })
                );
            }

            dispatch({type: SET_REGISTER_SOCIAL_STATE, payload: getAdvancedError('registration', errors[0])});
            metrika.send([SOCIAL, 'Ошибка регистрации социальщика']);
        });
};
