import {
    changeCaptchaState,
    setAuthMailError,
    setAuthMailCancelled,
    setPasswordError,
    changePagePopupVisibility,
    changeProcessedAccount
} from './actions';
import {SUCCESS_AUTH_LETTER} from './metrics_constants';
import metrics from '@blocks/metrics';
import reloadCaptcha from '@components/Captcha/actions/reloadCaptcha';
import switchToModeAddingAccount from './actions/switchToModeAddingAccount';
import {getRedirectUrl} from '@blocks/selectors';
import {ACCOUNT_TYPES, getAccountTypeByAlias} from '@blocks/authv2/utils/accountTypes';
import {amSetAnalyticsToRetpath} from '@blocks/authv2/actions/nativeMobileApi';

let store;
const mailAuthWrapper = {
    init(s) {
        this.setStore(s);
    },

    start(params) {
        this.restart(params);
    },

    restart() {
        const self = this;
        const state = this.getStore().getState();
        const {
            common: {track_id: trackId, csrf},
            am = {}
        } = state;

        self.track_id = trackId;
        self.csrf_token = csrf;
        self.retpath = getRedirectUrl(state);

        if (am.isAm) {
            this.getStore().dispatch(amSetAnalyticsToRetpath('magic_link_auth'));
            self.retpath = getRedirectUrl(this.getStore().getState());
        }

        self.trackNotFoundErrorCount = 0;

        self.stop();

        self._interval = 300;

        window.intervalTimeout = setTimeout(function() {
            self._interval = 1000;
        }, 60 * 1000);

        window.redirectTimeout = setTimeout(function() {
            window.location.reload(true);
        }, 60 * 1000 * 10);

        self._stopped = false;
        self.restartPolling();
    },

    restartPolling() {
        const self = this;

        if (!self._stopped) {
            setTimeout(function() {
                self.poll();
            }, self._interval);
        }
    },

    poll() {
        const self = this;
        const storeObj = this.getStore();
        const req = $.ajax('/auth/letter/status/', {
            data: {
                track_id: self.track_id,
                csrf_token: self.csrf_token,
                retpath: self.retpath
            },
            dataType: 'json',
            type: 'POST'
        });

        req.done(function(result) {
            const isAuthChallengeRequired = result.state && result.state === 'auth_challenge';

            if (isAuthChallengeRequired) {
                self.stop();

                window.location.href = `/auth/challenges/?track_id=${result.track_id}`;
                return;
            }

            if (result.status === 'ok' && result.magic_link_confirmed) {
                self.stop();
                metrics.send([SUCCESS_AUTH_LETTER]);

                let redirectUrl = `/auth/finish/?track_id=${result.track_id}`;

                if (self.retpath || result.retpath) {
                    redirectUrl += `&retpath=${encodeURIComponent(self.retpath || result.retpath)}`;
                }

                window.location.href = redirectUrl;
                return;
            }

            if (result.errors) {
                const state = storeObj.getState();
                const {
                    auth: {processedAccount}
                } = state;
                const {primaryAliasType} = processedAccount || {};
                const isMailAuth = getAccountTypeByAlias(primaryAliasType) === ACCOUNT_TYPES.LITE;
                const isInvalidationError = result.errors[0] === 'magic_link.invalidated';
                const error = isInvalidationError && !isMailAuth ? 'messenger_link.invalidated' : result.errors[0];
                const trackErrors = ['track.not_found', 'track_id.empty'];
                const isTrackError = trackErrors.includes(error);

                storeObj.dispatch(changePagePopupVisibility(false));

                if (!isTrackError) {
                    self.stop();
                    storeObj.dispatch(setAuthMailError(error));
                }

                if (error === 'captcha.required') {
                    storeObj.dispatch(setPasswordError(''));
                    storeObj.dispatch(changeCaptchaState(true));
                    storeObj.dispatch(setAuthMailCancelled(true));
                    storeObj.dispatch(reloadCaptcha());
                    return;
                }

                if (error === 'magic_link.invalidated' || error === 'messenger_link.invalidated') {
                    const state = storeObj.getState();
                    const {
                        mailAuth: {isUpdatedAuthLetterStatus},
                        auth: {processedAccount}
                    } = state;

                    if (processedAccount.allowed_auth_methods.length === 1) {
                        storeObj.dispatch(switchToModeAddingAccount());
                        return;
                    }

                    if (!isUpdatedAuthLetterStatus) {
                        const updatedAccount = Object.assign({}, processedAccount);

                        updatedAccount.allowed_auth_methods = updatedAccount.allowed_auth_methods.filter(
                            (method) => method !== 'magic_link'
                        );

                        storeObj.dispatch(changeProcessedAccount(updatedAccount));
                    }
                }

                if (isTrackError) {
                    if (self.trackNotFoundErrorCount > 5) {
                        self.stop();
                        storeObj.dispatch(setAuthMailError('global'));
                        return;
                    }

                    self.trackNotFoundErrorCount++;
                    self.restartPolling();
                }
            }

            self.restartPolling();
        });
    },

    setStore(s) {
        store = s;
    },

    getStore() {
        return store;
    },

    stopPoll() {
        this._stopped = true;
    },

    stop() {
        clearTimeout(window.intervalTimeout);
        clearTimeout(window.redirectTimeout);

        this.stopPoll();
    }
};

export default mailAuthWrapper;
