import React from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import mapStateToProps from './mapStateToProps';
import DefaultAccountListItem from './DefaultAccountListItem/DefaultAccountListItem.jsx';
import AuthorizedAccountListItem from './AuthorizedAccountListItem/AuthorizedAccountListItem.jsx';
import SuggestedAccountListItem from './SuggestedAccountListItem/SuggestedAccountListItem.jsx';
import InvalidAccountListItem from './InvalidAccountListItem/InvalidAccountListItem.jsx';
import AddAccountButton from './AddAccountButton/AddAccountButton.jsx';
import {ContextMenu} from './ContextMenu';

const b = cn('AuthAccountList');

const AccountList = ({
    authorizedAccounts,
    defaultAccount,
    hasUnauthorizedAccounts,
    suggestedAccounts,
    invalidAccounts,
    isSessionOverflow,
    useNewSuggestByPhone
}) => (
    <div className={b()} data-t='account'>
        <ul data-t='account-list' className={b('items')}>
            {defaultAccount && (
                <li className={b('itemBlock', {isNewSuggest: useNewSuggestByPhone})} data-t='account-list-item-box'>
                    <DefaultAccountListItem key={defaultAccount.uid} account={defaultAccount} />
                </li>
            )}

            {authorizedAccounts
                .filter((account) => !defaultAccount || account.uid !== defaultAccount.uid)
                .map((account, index) => (
                    <li
                        className={b('itemBlock', {isNewSuggest: useNewSuggestByPhone})}
                        key={`account-${index}`}
                        data-t='account-list-item-box'
                    >
                        <AuthorizedAccountListItem key={account.uid} account={account} />
                    </li>
                ))}
        </ul>

        {hasUnauthorizedAccounts && (
            <>
                <h2 className={b('unathorizedAccountsTitle', {isNewSuggest: useNewSuggestByPhone})}>
                    {i18n('_AUTH_.suggested_accounts_title')}
                </h2>

                <ul data-t='unathorized-account-list' className={b('items')}>
                    {suggestedAccounts.map((account, index) => (
                        <li
                            className={b('itemBlock', {isNewSuggest: useNewSuggestByPhone})}
                            key={`suggestedAccount-${index}`}
                            data-t='account-list-item-box'
                        >
                            <SuggestedAccountListItem key={account.uid} account={account} />
                        </li>
                    ))}

                    {invalidAccounts.map((account, index) => (
                        <li
                            className={b('itemBlock', {isNewSuggest: useNewSuggestByPhone})}
                            key={`invalidAccount-${index}`}
                            data-t='account-list-item-box'
                        >
                            <InvalidAccountListItem key={account.uid} account={account} />
                        </li>
                    ))}
                </ul>
            </>
        )}

        {!isSessionOverflow && <AddAccountButton />}
        {<ContextMenu />}
    </div>
);

AccountList.propTypes = {
    authorizedAccounts: PropTypes.array.isRequired,
    defaultAccount: PropTypes.object,
    suggestedAccounts: PropTypes.array.isRequired,
    invalidAccounts: PropTypes.array.isRequired,
    hasUnauthorizedAccounts: PropTypes.bool.isRequired,
    isSessionOverflow: PropTypes.bool.isRequired,
    useNewSuggestByPhone: PropTypes.bool
};

export default connect(mapStateToProps)(AccountList);
