import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';

import {Field} from '@components/Field';
import CaptchaField from '../CaptchaField/CaptchaField.jsx';
import {Button} from '@components/Button';
import PhoneConfirmationTimer from '../PhoneConfirmationTimer/PhoneConfirmationTimer.jsx';
import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import errorMessages from '../../errors';
import submitPhoneConfirmationCode from '../../actions/submitPhoneConfirmationCode';
import requestPhoneConfirmationCode from '../../actions/requestPhoneConfirmationCode';
import skipAdditionalData from '../../actions/skipAdditionalData';
import Title from '../Title/Title.jsx';
import PagePopupContent from '../PagePopupContent/PagePopupContent.jsx';
import Form from '../Form/Form.jsx';
import {checkDigitsAndDash} from '@blocks/utils';

class AdditionalDataRequestPhoneConfirmationPopup extends Component {
    constructor(props) {
        super(props);

        this.handleConfirmationCodeChange = this.handleConfirmationCodeChange.bind(this);
        this.approvePhone = this.approvePhone.bind(this);
        this.sendConfirmationCode = this.sendConfirmationCode.bind(this);
        this.goBack = this.goBack.bind(this);
        this.startTimer = this.startTimer.bind(this);
        this.stopTimer = this.stopTimer.bind(this);

        this.timerInterval = null;
        this.state = {currentTimestamp: Number(new Date())};
    }

    componentDidMount() {
        this.stopTimer();
        this.startTimer();
    }

    componentWillUpdate(nextProps) {
        if (this.props.denyResendUntil !== nextProps.denyResendUntil) {
            this.setState({currentTimestamp: Number(new Date())});
            this.startTimer();
        }
    }

    componentWillUnmount() {
        this.handleConfirmationCodeChange('');
        this.props.changeNativeInputValue('confirmation_code', '');
        this.props.clearErrors();
        this.stopTimer();
    }

    handleConfirmationCodeChange(value) {
        const {changeConfirmationCode} = this.props;
        const isValid = checkDigitsAndDash(value);

        if (isValid) {
            changeConfirmationCode(value);
        }
    }

    approvePhone(event) {
        event.preventDefault();

        const {sendApiRequest, confirmationCode} = this.props;

        sendApiRequest(submitPhoneConfirmationCode, confirmationCode);
    }

    sendConfirmationCode(event) {
        event.preventDefault();

        const {sendApiRequest, phoneNumber} = this.props;

        sendApiRequest(requestPhoneConfirmationCode, phoneNumber);
    }

    goBack() {
        const {sendApiRequest, sendAdditionalDataRequestMetrics} = this.props;

        sendAdditionalDataRequestMetrics('skip_phone', null, 'Отложить');
        sendApiRequest(skipAdditionalData);
    }

    startTimer() {
        this.timerInterval = setInterval(() => {
            const currentTimestamp = Number(new Date());

            this.setState({currentTimestamp});

            if (currentTimestamp >= Number(this.props.denyResendUntil)) {
                this.stopTimer();
            }
        }, 1000);
    }

    stopTimer() {
        if (this.timerInterval) {
            clearInterval(this.timerInterval);
            this.timerInterval = null;
        }
    }

    getSendAgainButtonText(showTimer) {
        const {denyResendUntil} = this.props;
        const {currentTimestamp} = this.state;

        if (showTimer) {
            return (
                <span>
                    {i18n('_AUTH_.enter-confirmation-code.send-again.v2')}
                    {'  '}
                    <PhoneConfirmationTimer startTime={currentTimestamp} endTime={Number(denyResendUntil)} />
                </span>
            );
        }

        return i18n('_AUTH_.enter-confirmation-code.send-again.v2');
    }

    render() {
        const {
            confirmationCode,
            isCaptchaRequired,
            denyResendUntil,
            popupTitle,
            errors,
            isWithCodeFormatExp
        } = this.props;
        const {currentTimestamp} = this.state;
        const showTimer = currentTimestamp + 1000 < Number(denyResendUntil); // не показывать нулевой таймер

        let fieldError = null;

        if (errors.length) {
            fieldError = errorMessages[errors[0]] ? errorMessages[errors[0]] : errorMessages.internal;
        }

        const sendAgainButtonText = this.getSendAgainButtonText(showTimer);

        return (
            <PagePopupContent cls='phone-confirmation' dataT='additional-data-request-phone-confirmation'>
                <Title>
                    <span dangerouslySetInnerHTML={{__html: popupTitle}} />
                </Title>

                <Form cls='phone-confirmation' dataT='additional-data-request-phone-confirmation'>
                    <Field
                        size='l'
                        label={i18n('_AUTH_.phone.enter-confirmation-code.label')}
                        value={confirmationCode}
                        name='confirmation_code'
                        type='text'
                        error={fieldError}
                        maxLength={isWithCodeFormatExp ? 7 : 6}
                        onChange={this.handleConfirmationCodeChange}
                        view='big-input'
                    />

                    {isCaptchaRequired && <CaptchaField />}

                    <div data-t='phone_enter_confirmation_code_submit' className='passp-button'>
                        <Button
                            type='submit'
                            text={i18n('_AUTH_.phone.enter-confirmation-code.submit')}
                            view='action'
                            size='l'
                            width='max'
                            onClick={this.approvePhone}
                        />
                    </div>

                    <div data-t='phone_send_again' className='passp-button'>
                        <Button
                            type='button'
                            text={sendAgainButtonText}
                            view='pseudo'
                            size='l'
                            width='max'
                            disabled={showTimer}
                            onClick={this.sendConfirmationCode}
                        />
                    </div>

                    <div data-t='phone_enter_confirmation_code_skip' className='passp-button'>
                        <Button
                            type='button'
                            text={i18n('_AUTH_.phone.enter-confirmation-code.skip')}
                            view='pseudo'
                            size='l'
                            width='max'
                            onClick={this.goBack}
                        />
                    </div>
                </Form>
            </PagePopupContent>
        );
    }
}

AdditionalDataRequestPhoneConfirmationPopup.propTypes = {
    confirmationCode: PropTypes.string,
    phoneNumber: PropTypes.string,
    popupTitle: PropTypes.string,
    isCaptchaRequired: PropTypes.bool,
    denyResendUntil: PropTypes.number,
    errors: PropTypes.array,
    changeConfirmationCode: PropTypes.func.isRequired,
    sendAdditionalDataRequestMetrics: PropTypes.func.isRequired,
    sendApiRequest: PropTypes.func.isRequired,
    changeNativeInputValue: PropTypes.func.isRequired,
    clearErrors: PropTypes.func.isRequired,
    isWithCodeFormatExp: PropTypes.bool
};

export default connect(mapStateToProps, mapDispatchToProps)(AdditionalDataRequestPhoneConfirmationPopup);
