import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {CSSTransition, TransitionGroup} from 'react-transition-group';

import mapStateToProps from './mapStateToProps';

const ANIMATION_TIMEOUTS = {enter: 300, exit: 300};

class AnimatedLayer extends Component {
    constructor(props) {
        super(props);

        this.state = {
            transitionClassName: '',
            historyKeys: []
        };
    }

    shouldComponentUpdate(nextProps, nextState) {
        const {location: currentLocation, isLoading: currentLoading} = this.props;
        const {location: nextLocation, history: nextHistory, isLoading: nextLoading} = nextProps;
        const {transitionClassName, historyKeys} = this.state;

        const currentPath = currentLocation.pathname === '/auth' ? '/auth/add' : currentLocation.pathname;
        const nextPath = nextLocation.pathname === '/auth' ? '/auth/add' : nextLocation.pathname;
        const currentSearch = currentLocation.search;
        const nextSearch = nextLocation.search;

        if (transitionClassName !== nextState.transitionClassName) {
            return true;
        }

        if (historyKeys.length !== nextState.historyKeys.length) {
            return true;
        }

        if (currentLoading !== nextLoading) {
            return true;
        }

        if (currentPath === nextPath && currentSearch === nextSearch) {
            return false;
        }

        return nextHistory.action.toLowerCase() !== 'replace';
    }

    componentDidUpdate(prevProps) {
        const {location} = prevProps;
        const {historyKeys} = this.state;
        const {location: nextLocation, history: nextHistory} = this.props;

        if (location === nextLocation) {
            return;
        }

        if (location.pathname === nextLocation.pathname) {
            return;
        }

        if (nextLocation.state && nextLocation.state.isPreviousStep) {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState((previousState) => ({
                transitionClassName: 'passp-route-back',
                historyKeys: previousState.historyKeys.concat([nextLocation.key])
            }));
            return;
        }

        if (nextHistory.action.toLowerCase() === 'push') {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState((previousState) => ({
                transitionClassName: 'passp-route-forward',
                historyKeys: previousState.historyKeys.concat([nextLocation.key])
            }));
            return;
        }

        const lastHistoryIndex = historyKeys.length - 1;
        const previousHistoryKey = historyKeys[lastHistoryIndex - 1];

        if (nextLocation.key === previousHistoryKey) {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState((previousState) => ({
                transitionClassName: 'passp-route-back',
                historyKeys: previousState.historyKeys.slice(0, -1)
            }));
            return;
        }

        // eslint-disable-next-line react/no-did-update-set-state
        this.setState((previousState) => ({
            transitionClassName: 'passp-route-forward',
            historyKeys: previousState.historyKeys.concat([nextLocation.key])
        }));
    }

    render() {
        const {location} = this.props;
        const {transitionClassName} = this.state;

        return (
            <TransitionGroup className={transitionClassName}>
                <CSSTransition key={location.key} timeout={ANIMATION_TIMEOUTS} classNames='passp-route' appear={true}>
                    <div className='passp-route'>{this.props.children}</div>
                </CSSTransition>
            </TransitionGroup>
        );
    }
}

AnimatedLayer.propTypes = {
    location: PropTypes.object,
    history: PropTypes.object,
    accounts: PropTypes.object,
    magicError: PropTypes.string,
    loginError: PropTypes.string,
    passwordError: PropTypes.string,
    captchaError: PropTypes.string,
    additionalDataErrors: PropTypes.array,
    isShowMobileMenu: PropTypes.bool,
    isLoading: PropTypes.bool,
    isCaptchaRequired: PropTypes.bool,
    isCanRegister: PropTypes.bool,
    children: PropTypes.node.isRequired
};

export default connect(mapStateToProps)(AnimatedLayer);
