import React, {Component} from 'react';
import {connect} from 'react-redux';
import ReactDOM from 'react-dom';
import PropTypes from 'prop-types';

import {changeCaptchaState, updateCaptchaValue} from '@blocks/authv2/actions';
import {amSetAnalyticsToRetpath} from '@blocks/authv2/actions/nativeMobileApi';
import mapStateToProps from './mapStateToProps';

import {Field} from '@components/Field';
import {Captcha} from '@components/Captcha';

class CaptchaField extends Component {
    constructor(props) {
        super(props);

        this.onChangeCaptcha = this.onChangeCaptcha.bind(this);
        this.focusOnCaptchaField = this.focusOnCaptchaField.bind(this);
    }

    componentDidMount() {
        const {isCaptchaRequired, isAm, dispatch} = this.props;

        if (isAm) {
            dispatch(amSetAnalyticsToRetpath('captcha'));
        }

        if (isCaptchaRequired) {
            this.focusOnCaptchaField();
        }
    }

    componentWillReceiveProps(nextProps) {
        const {isCaptchaRequired} = this.props;

        if (isCaptchaRequired !== nextProps.isCaptchaRequired && nextProps.isCaptchaRequired) {
            setTimeout(() => {
                this.focusOnCaptchaField();
            }, 0);
        }
    }

    componentWillUnmount() {
        const {dispatch} = this.props;

        dispatch(changeCaptchaState(false));
        dispatch(updateCaptchaValue(''));
    }

    onChangeCaptcha(value) {
        const {dispatch} = this.props;

        dispatch(updateCaptchaValue(value));
    }

    focusOnCaptchaField() {
        if (this.props.isNoTabletTouch) {
            return;
        }

        // eslint-disable-next-line react/no-find-dom-node
        const fieldInput = ReactDOM.findDOMNode(this.captchaField).querySelector('input');

        fieldInput.focus();
        fieldInput.click();
    }

    render() {
        const {
            isCaptchaRequired,
            isShowCaptcha,
            introSound,
            captchaSound,
            imageUrl,
            loading,
            loadingAudio,
            playing,
            type,
            captchaKey,
            captchaAnswer,
            lang,
            fieldError,
            dispatch,
            forcedError,
            env
        } = this.props;

        if (!isCaptchaRequired && !isShowCaptcha) {
            return null;
        }

        return (
            <div className={`passp-captcha-field`}>
                <Captcha
                    isMobile={false}
                    lang={lang}
                    introSound={introSound}
                    captchaSound={captchaSound}
                    imageUrl={imageUrl}
                    env={env}
                    captchaKey={captchaKey}
                    loading={loading}
                    loadingAudio={loadingAudio}
                    playing={playing}
                    type={type}
                    dispatch={dispatch}
                    visible={true}
                />

                <input type='hidden' name='captcha_key' value={captchaKey || ''} />

                <Field
                    size='l'
                    ref={(field) => {
                        this.captchaField = field;
                    }}
                    label={i18n('_AUTH_.enter_captcha_label')}
                    value={captchaAnswer}
                    name='captcha_answer'
                    type='text'
                    error={fieldError || forcedError}
                    link={null}
                    onChange={this.onChangeCaptcha}
                    view='big-input'
                />
            </div>
        );
    }
}

CaptchaField.defaultProps = {
    captcha: {
        captchaKey: ''
    },
    isShowCaptcha: false
};

CaptchaField.propTypes = {
    isCaptchaRequired: PropTypes.bool.isRequired,
    isShowCaptcha: PropTypes.bool.isRequired,
    introSound: PropTypes.string,
    captchaSound: PropTypes.string,
    imageUrl: PropTypes.string,
    env: PropTypes.shape({
        type: PropTypes.string
    }),
    loading: PropTypes.bool,
    loadingAudio: PropTypes.bool,
    playing: PropTypes.bool,
    type: PropTypes.string,
    captchaKey: PropTypes.string,
    captchaAnswer: PropTypes.string.isRequired,
    lang: PropTypes.string.isRequired,
    fieldError: PropTypes.string,
    forcedError: PropTypes.string,
    dispatch: PropTypes.func.isRequired,
    isNoTabletTouch: PropTypes.bool,
    isAm: PropTypes.bool
};

export default connect(mapStateToProps)(CaptchaField);
