import React, {Fragment, Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import mapStateToProps from './mapStateToProps';
import {Button} from '@components/Button';
import Title from '../../Title/Title.jsx';
import Form from '../../Form/Form.jsx';
import {Field} from '@components/Field';
import PagePopupContent from '../../PagePopupContent/PagePopupContent.jsx';
import Timer from '../../RestoreLogin/Timer/Timer.jsx';
import metrics from '../../../../metrics';
import {CHALLENGES_PAGE, CHALLENGES_PAGE_PHONE_CALL_POPUP} from '../../../metrics_constants';
import {setChallengeData, updateFieldValue} from '../../../actions/challenge';
import checkCode from '../../../actions/challenge/checkCode';
import validatePhone from '../../../actions/challenge/validatePhone';
import confirmPhone from '../../../actions/challenge/confirmPhone';
import {FIELDS_NAMES} from '@components/Field/names';
import {checkDigitsAndDash, getFirstSymbolForQuery} from '@blocks/utils';
import {formatLocTextWithLink} from '@blocks/authv2/utils/formatLocTextWithLink';
import {CodeField} from '@components/CodeField';

const SUPPORT_FORM_LINK = '/restoration/form?unconditional=true';

class CallPopup extends Component {
    constructor(props) {
        super(props);

        this.state = {
            isRepeatActionBtnDisabled: true
        };
    }

    ref = React.createRef();

    focus = () => {
        if (this.ref.current && this.ref.current.focus) {
            this.ref.current.focus();
        }
    };

    componentDidMount() {
        setTimeout(this.focus, 300);

        metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_PHONE_CALL_POPUP]);
    }

    componentWillUnmount() {
        this.props.dispatch(
            setChallengeData({
                errorText: '',
                errorCode: '',
                fieldValue: ''
            })
        );
    }

    updateFieldValue = (value) => {
        const isValid = checkDigitsAndDash(value);

        if (isValid) {
            this.props.dispatch(updateFieldValue(value));
        }
    };

    sendSms = () => {
        const {dispatch} = this.props;

        dispatch(
            setChallengeData({
                isConfirmWithCall: false
            })
        );
        dispatch(confirmPhone());
        this.updateRepeatBtnAction();
    };

    callAgain = () => {
        const {dispatch} = this.props;

        dispatch(validatePhone());
        this.updateRepeatBtnAction();
    };

    submitForm = (event) => {
        event.preventDefault();
        this.props.dispatch(checkCode());
    };

    updateRepeatBtnAction = () => {
        this.setState((prevState) => ({
            isRepeatActionBtnDisabled: !prevState.isRepeatActionBtnDisabled
        }));
    };

    getSendAgainButtonText = (isCallAgain) => {
        const {btnText, smsBtnText} = this.props;

        return (
            <Fragment>
                {isCallAgain ? btnText : smsBtnText}
                {this.state.isRepeatActionBtnDisabled && <Timer onTimerEnd={this.updateRepeatBtnAction} />}
            </Fragment>
        );
    };

    render() {
        const {
            title,
            errorText,
            value,
            maxFieldLength,
            isOneStepChallenge,
            hasRoundViewExp,
            isWhiteLabel,
            isAm,
            appId,
            isConfirmWithCall,
            challengeType
        } = this.props;
        const isCallAgain = isConfirmWithCall && challengeType === 'dictation';
        const buttonText = this.getSendAgainButtonText(isCallAgain);
        const modifiedSupportFormLink =
            isAm && appId
                ? `${SUPPORT_FORM_LINK}${getFirstSymbolForQuery(SUPPORT_FORM_LINK)}app_id=${appId}`
                : SUPPORT_FORM_LINK;

        return (
            <PagePopupContent cls='restore-login-enter-confirmation-code' dataT='challenges-call'>
                <Title>
                    <div dangerouslySetInnerHTML={{__html: title}} />
                </Title>

                <Form onSubmit={this.submitForm} cls='enter-confirmation-code' dataT='challenges-call'>
                    {hasRoundViewExp ? (
                        <CodeField codeLength={maxFieldLength} error={errorText} />
                    ) : (
                        <Field
                            size='l'
                            ref={this.ref}
                            label=''
                            value={value}
                            name={FIELDS_NAMES.PHONE_CODE}
                            type='tel'
                            error={errorText}
                            options={{maxLength: maxFieldLength}}
                            onChange={this.updateFieldValue}
                            view='big-input'
                        />
                    )}

                    <div data-t='submit-phone-code-btn' className='passp-button'>
                        <Button
                            text={i18n('_AUTH_.RestoreLogin.next-step')}
                            type='submit'
                            view='action'
                            size='l'
                            width='max'
                        />
                    </div>

                    <div data-t='send-code-again-btn' className='passp-button passp-send-again-btn challenge-submit'>
                        <Button
                            text={buttonText}
                            type='button'
                            view='pseudo'
                            size='l'
                            width='max'
                            disabled={this.state.isRepeatActionBtnDisabled}
                            onClick={isCallAgain ? this.callAgain : this.sendSms}
                        />
                    </div>
                </Form>
                {isOneStepChallenge && !isWhiteLabel && (
                    <div
                        className='auth-challenge-note'
                        dangerouslySetInnerHTML={{
                            __html: formatLocTextWithLink(
                                i18n('_AUTH_.challenges.sms2fa-advice-note'),
                                modifiedSupportFormLink
                            )
                        }}
                    />
                )}
            </PagePopupContent>
        );
    }
}

export default connect(mapStateToProps)(CallPopup);

CallPopup.propTypes = {
    isConfirmWithCall: PropTypes.bool,
    errorText: PropTypes.string,
    title: PropTypes.string,
    btnText: PropTypes.string,
    smsBtnText: PropTypes.string,
    value: PropTypes.string,
    maxFieldLength: PropTypes.number,
    isWhiteLabel: PropTypes.bool,
    isOneStepChallenge: PropTypes.bool,
    hasRoundViewExp: PropTypes.bool,
    dispatch: PropTypes.func.isRequired,
    isAm: PropTypes.bool,
    appId: PropTypes.string,
    challengeType: PropTypes.string
};
