import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import TrustSDK from '@yandex-trust-sdk/base';
import {classNames} from '@blocks/utils';
import {Button} from '@components/Button';
import {LoaderId} from '@components/LoaderId';
import metrics from '@blocks/metrics';
import {
    CHALLENGES_PAGE,
    CHALLENGES_PAGE_3DS_SHOW,
    CHALLENGES_PAGE_3DS_GO_TO_3DS_URL,
    CHALLENGES_PAGE_3DS_TIMEOUT_ERROR,
    CHALLENGES_PAGE_3DS_ERROR,
    CHALLENGES_PAGE_3DS_ABORT,
    CHALLENGES_PAGE_3DS_SUCCESS,
    CHALLENGES_PAGE_3DS_RETURN_TO_SERVICE
} from '@blocks/authv2/metrics_constants';
import api from '@blocks/api';

const defaultCssClass = classNames('auth-challenge__3ds');
const ERROR_TIMEOUT = 10000;

class ChallengeWith3ds extends PureComponent {
    state = {
        trustForm: null,
        isTrustFrameReady: true,
        isFullscreenLoading: false
    };

    componentDidMount() {
        const {isOneStep3dsExp} = this.props;

        if (isOneStep3dsExp) {
            this.handleSubmit();
        }

        metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_3DS_SHOW]);
    }

    componentDidUpdate(prevProps) {
        const {url3ds} = this.props;

        if (url3ds && url3ds !== prevProps.url3ds) {
            this.createTrustFrame();
        }
    }

    componentWillUnmount() {
        clearTimeout(this.errorTimeout);
        if (this.state.trustForm) {
            this.umnountTrustFrame();
        }
    }

    createTrustFrame = () => {
        const {url3ds, isOneStep3dsExp} = this.props;

        const sdk = TrustSDK.create();

        this.errorTimeout = setTimeout(this.onTimeoutError, ERROR_TIMEOUT);

        const sdkFormPromise = sdk.paymentForm({url: url3ds});

        sdkFormPromise.then((paymentFrame) => {
            this.setState({trustForm: paymentFrame});

            this.addFrameMessageListener();

            this.state.trustForm.mount('#trustFrameParent', {
                className: isOneStep3dsExp ? defaultCssClass('trust-frame') : defaultCssClass('trust-frame-full'),
                title: '3ds'
            });
        });
    };

    onTimeoutError = () => {
        const {setChallengeError, set3dsLoadingStatus} = this.props;

        metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_3DS_TIMEOUT_ERROR]);
        setChallengeError({
            code: 'internal3ds',
            text: i18n('_AUTH_.challenge.3ds-error.internal')
        });
        if (this.state.trustForm) {
            this.umnountTrustFrame();
        }
        set3dsLoadingStatus(false);
        api.log('Trust form on chaas 3ds timeout error');
    };

    frameMessageListener = (event) => {
        const {trustForm: {frameOrigin} = {}} = this.state;

        if (event.origin !== frameOrigin) {
            return;
        }

        let eventData = event.data || '';

        if (typeof eventData === 'string') {
            try {
                eventData = JSON.parse(event.data);
            } catch (error) {
                api.log(`Trust form on chaas 3ds event parsing error with '${error}'`);
            }
        }

        let logString = `Trust form on chaas 3ds responded with event '${JSON.stringify(
            eventData.type || eventData.data || eventData
        )}'`;

        if (eventData.type === '3ds-status' && eventData.data) {
            logString = `${logString} - ${JSON.stringify(eventData.data)}`;
        }

        api.log(logString);

        if (eventData.type === '3ds-status' && eventData.data) {
            this.onEvent(eventData.data);
        } else if (eventData.type) {
            this.onEvent(eventData.type);
        }
    };

    addFrameMessageListener = () => {
        window.addEventListener('message', this.frameMessageListener, false);
    };

    removeFrameMessageListener = () => {
        window.removeEventListener('message', this.frameMessageListener);
    };

    onEvent = (type) => {
        switch (type) {
            case 'init-checkout':
                this.onReadyTrustFrame();
                metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_3DS_GO_TO_3DS_URL]);
                break;
            case 'auth-end':
                metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_3DS_SUCCESS]);
                this.on3dsFinish();
                break;
            case 'abort':
                metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_3DS_ABORT]);
                this.on3dsFinish();
                break;
            case 'error':
            case '3ds-failure':
                metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_3DS_ERROR]);
                this.on3dsFinish();
                break;
        }
    };

    onReadyTrustFrame = () => {
        const {set3dsLoadingStatus, setTrustFrameOpenState} = this.props;

        clearTimeout(this.errorTimeout);
        set3dsLoadingStatus(false);
        setTrustFrameOpenState(true);
        this.setState({isTrustFrameReady: true});
    };

    on3dsFinish = () => {
        const {commit3ds, isOneStep3dsExp} = this.props;

        commit3ds();
        !isOneStep3dsExp && this.setState({isFullscreenLoading: true});
        this.umnountTrustFrame();
    };

    umnountTrustFrame = () => {
        const {trustForm} = this.state;
        const {setTrustFrameOpenState} = this.props;

        if (trustForm) {
            trustForm.unmount();
            setTrustFrameOpenState(false);
            this.removeFrameMessageListener();
        }
    };

    handleSubmit = () => {
        const {submit3ds} = this.props;

        this.setState({isTrustFrameReady: false});

        submit3ds();
    };

    onReturnToServiceClick = () => {
        const {redirectToErrorFrom3ds} = this.props;

        metrics.send([CHALLENGES_PAGE, CHALLENGES_PAGE_3DS_RETURN_TO_SERVICE]);
        redirectToErrorFrom3ds();
    };

    renderError = () => {
        const {errorText} = this.props;

        return (
            <div className={defaultCssClass('error')}>
                <div className={defaultCssClass('error-text')}>{errorText}</div>
                <Button
                    onClick={this.onReturnToServiceClick}
                    text={i18n('_AUTH_.go_back')}
                    view='action'
                    size='l'
                    width='max'
                />
            </div>
        );
    };

    render() {
        const {isTrustFrameReady, isFullscreenLoading} = this.state;
        const {isOneStep3dsExp, is3dsLoading, errorText, isTrustFrameOpened} = this.props;

        return (
            <div className={defaultCssClass()}>
                {is3dsLoading ? (
                    <div
                        className={
                            isFullscreenLoading
                                ? defaultCssClass('loader-wrapper-fullscreen')
                                : defaultCssClass('loader-wrapper')
                        }
                    >
                        <LoaderId />
                    </div>
                ) : (
                    <>
                        {!isOneStep3dsExp && !errorText && (
                            <form className={defaultCssClass('form')}>
                                <div data-t='challenge_sumbit_3ds' className='passp-button'>
                                    <Button
                                        onClick={this.handleSubmit}
                                        type='button'
                                        text={i18n('_AUTH_.confirm_code')}
                                        view='action'
                                        size='l'
                                        width='max'
                                        progress={!isTrustFrameReady}
                                    />
                                </div>
                            </form>
                        )}
                        {errorText && this.renderError()}
                    </>
                )}

                <div
                    style={isTrustFrameOpened ? {display: 'flex'} : {display: 'none'}}
                    className={
                        isOneStep3dsExp
                            ? defaultCssClass('trust-frame-parent')
                            : defaultCssClass('trust-frame-parent-fullscreen')
                    }
                    id='trustFrameParent'
                />
            </div>
        );
    }
}

export default ChallengeWith3ds;

ChallengeWith3ds.propTypes = {
    commit3ds: PropTypes.func.isRequired,
    submit3ds: PropTypes.func.isRequired,
    url3ds: PropTypes.string.isRequired,
    redirectToErrorFrom3ds: PropTypes.func.isRequired,
    isOneStep3dsExp: PropTypes.bool.isRequired,
    is3dsLoading: PropTypes.bool.isRequired,
    set3dsLoadingStatus: PropTypes.func.isRequired,
    errorText: PropTypes.string,
    setTrustFrameOpenState: PropTypes.func.isRequired,
    isTrustFrameOpened: PropTypes.bool,
    setChallengeError: PropTypes.func.isRequired
};
