import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';

import escape from 'lodash/escape';

import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import {Field} from '@components/Field';
import CaptchaField from '../CaptchaField/CaptchaField.jsx';
import {Button} from '@components/Button';
import Title from '../Title/Title.jsx';
import Form from '../Form/Form.jsx';
import PagePopupContent from '../PagePopupContent/PagePopupContent.jsx';
import submitEmailConfirmationCode from '../../actions/submitEmailConfirmationCode';
import requestEmailConfirmationCode from '../../actions/requestEmailConfirmationCode';
import skipAdditionalData from '../../actions/skipAdditionalData';
import errorMessages from '../../errors';

class EmailConfirmationPopup extends Component {
    constructor(props) {
        super(props);

        this.handleConfirmationCodeChange = this.handleConfirmationCodeChange.bind(this);
        this.approveEmail = this.approveEmail.bind(this);
        this.sentConfirmationCode = this.sentConfirmationCode.bind(this);
        this.goBack = this.goBack.bind(this);
    }

    componentDidMount() {
        const {sendAdditionalDataRequestMetrics} = this.props;

        sendAdditionalDataRequestMetrics('show_email', null, 'Показ');
    }

    componentWillUnmount() {
        this.props.clearErrors();
        this.handleConfirmationCodeChange('');
        this.props.changeNativeInputValue('confirmation_code', '');
    }

    handleConfirmationCodeChange(value) {
        const {changeConfirmationCode} = this.props;

        changeConfirmationCode(value);
    }

    approveEmail(event) {
        event.preventDefault();

        const {sendApiRequest, confirmationCode} = this.props;

        sendApiRequest(submitEmailConfirmationCode, confirmationCode);
    }

    sentConfirmationCode(event) {
        event.preventDefault();

        const {sendApiRequest, email} = this.props;

        sendApiRequest(requestEmailConfirmationCode, email);
    }

    goBack() {
        const {sendApiRequest, sendAdditionalDataRequestMetrics} = this.props;

        sendAdditionalDataRequestMetrics('skip_email', null, 'Отложить');
        sendApiRequest(skipAdditionalData);
    }

    render() {
        const {confirmationCode, isCaptchaRequired, errors, email} = this.props;

        let fieldError = null;

        if (errors.length) {
            fieldError = errorMessages[errors[0]] ? errorMessages[errors[0]] : errorMessages.internal;
        }

        return (
            <PagePopupContent cls='email-confirmation' dataT='email-confirmation'>
                <Title>
                    <span
                        dangerouslySetInnerHTML={{
                            __html: i18n('_AUTH_.email.enter-confirmation-code.title.v2').replace(
                                '%1',
                                `<b>${escape(email)}</b>`
                            )
                        }}
                    />
                </Title>

                <Form cls='email-confirmation' dataT='email-confirmation'>
                    <Field
                        size='l'
                        label={i18n('_AUTH_.email.enter-confirmation-code.label')}
                        value={confirmationCode}
                        name='confirmation_code'
                        type='text'
                        error={fieldError}
                        onChange={this.handleConfirmationCodeChange}
                        view='big-input'
                    />

                    {isCaptchaRequired && <CaptchaField />}

                    <div data-t='email_enter_confirmation_code_submit' className='passp-button'>
                        <Button
                            type='submit'
                            text={i18n('_AUTH_.email.enter-confirmation-code.submit')}
                            view='action'
                            size='l'
                            width='max'
                            onClick={this.approveEmail}
                        />
                    </div>

                    <div data-t='email_send_again' className='passp-button'>
                        <Button
                            type='button'
                            text={i18n('_AUTH_.enter-confirmation-code.send-again.v2')}
                            view='pseudo'
                            size='l'
                            width='max'
                            onClick={this.sentConfirmationCode}
                        />
                    </div>

                    <div data-t='email_enter_confirmation_code_skip' className='passp-button'>
                        <Button
                            type='button'
                            text={i18n('_AUTH_.email.enter-confirmation-code.skip')}
                            view='pseudo'
                            size='l'
                            width='max'
                            onClick={this.goBack}
                        />
                    </div>
                </Form>
            </PagePopupContent>
        );
    }
}

EmailConfirmationPopup.propTypes = {
    confirmationCode: PropTypes.string,
    email: PropTypes.string,
    isCaptchaRequired: PropTypes.bool,
    errors: PropTypes.array,
    sendAdditionalDataRequestMetrics: PropTypes.func.isRequired,
    changeConfirmationCode: PropTypes.func.isRequired,
    changeNativeInputValue: PropTypes.func.isRequired,
    sendApiRequest: PropTypes.func.isRequired,
    clearErrors: PropTypes.func.isRequired
};

export default connect(mapStateToProps, mapDispatchToProps)(EmailConfirmationPopup);
