import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Link} from '@components/Link';
import metrics from '@blocks/metrics';
import mapStateToProps from './mapStateToProps';

const b = cn('AuthFooter');

class Footer extends Component {
    constructor(props) {
        super(props);

        this.toggleLangswitcherPopup = this.toggleLangswitcherPopup.bind(this);
        this.onOutsideClick = this.onOutsideClick.bind(this);

        this.state = {langPopupOpen: false};
    }

    componentDidMount() {
        if (window.addEventListener) {
            window.addEventListener('click', this.onOutsideClick);
        } else {
            window.attachEvent('onclick', this.onOutsideClick);
        }
    }

    componentWillUnmount() {
        if (window.removeEventListener) {
            window.removeEventListener('click', this.onOutsideClick);
        } else {
            window.detachEvent('onclick', this.onOutsideClick);
        }
    }

    toggleLangswitcherPopup(event) {
        event.stopPropagation();

        this.setState((currentState) => ({
            langPopupOpen: !currentState.langPopupOpen
        }));
    }

    onOutsideClick() {
        this.setState({
            langPopupOpen: false
        });
    }

    static handleEnterKey(event) {
        if (event.keyCode === 13) {
            event.target.click();
        }
    }

    render() {
        const {tld, lang, options, currentYear, helpLink, hasSwitcher, isWebView, isMobile} = this.props;
        const {langPopupOpen} = this.state;

        return (
            <footer className={b()}>
                <div className={b('wrapper')}>
                    {!isWebView && !isMobile && (
                        <div className={b('incognitoLink')}>
                            <span className={b('item')}>
                                {(['ru', 'com', 'com.tr', 'ua', 'by', 'kz', 'uz'].includes(tld) && (
                                    <Link
                                        view='default'
                                        onClick={() => {
                                            metrics.send(['Нажатие на ссылку инкогнито в футере']);
                                        }}
                                        target='_blank'
                                        href={`https://yandex.${tld}/support/common/browsers-settings/incognito.html`}
                                    >
                                        {i18n('_AUTH_.incognito_link')}
                                    </Link>
                                )) ||
                                    i18n('_AUTH_.incognito_link')}
                            </span>
                        </div>
                    )}

                    <div
                        className={b('mainBlock', {
                            single: isWebView || isMobile
                        })}
                    >
                        {hasSwitcher && (
                            <span onKeyUp={Footer.handleEnterKey} className={b('item')} aria-expanded={langPopupOpen}>
                                <Link onClick={this.toggleLangswitcherPopup} pseudo={true} className={b('itemCap')}>
                                    {lang}
                                </Link>
                                <span
                                    className={b('langList', {
                                        opened: langPopupOpen,
                                        closed: !langPopupOpen
                                    })}
                                >
                                    <ul>
                                        {options.map((option, optionIndex) => (
                                            <li className={b('langListItem')} key={optionIndex}>
                                                <Link href={option.url}>{option.text}</Link>
                                            </li>
                                        ))}
                                    </ul>
                                </span>
                            </span>
                        )}

                        {!isWebView && (
                            <span className={b('item')}>
                                <Link href={helpLink} target='_blank'>
                                    {i18n('_AUTH_.footer.help')}
                                </Link>
                            </span>
                        )}

                        <span className={b('item')}>
                            <span className={b('copyright')}>{`© ${tld === 'ru' ? '2001-' : ''}${currentYear}, `}</span>
                            {!isWebView ? (
                                <Link href={`//yandex.${tld}`}>{i18n('_AUTH_.yandex')}</Link>
                            ) : (
                                i18n('_AUTH_.yandex')
                            )}
                        </span>
                    </div>
                </div>
            </footer>
        );
    }
}

Footer.propTypes = {
    tld: PropTypes.string.isRequired,
    lang: PropTypes.string.isRequired,
    options: PropTypes.array.isRequired,
    helpLink: PropTypes.string.isRequired,
    currentYear: PropTypes.number.isRequired,
    hasSwitcher: PropTypes.bool.isRequired,
    isWebView: PropTypes.bool,
    isMobile: PropTypes.bool
};

export default connect(mapStateToProps)(Footer);
