import React, {Component} from 'react';
import {connect} from 'react-redux';
import classnames from 'classnames';
import PropTypes from 'prop-types';

import sendApiRequest from '../../actions/sendApiRequest';
import multiStepAuthStart from '../../actions/multiStepAuthStart';
import multiStepAuthCommitPassword from '../../actions/multiStepAuthCommitPassword';
import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';

import CaptchaField from '../CaptchaField/CaptchaField.jsx';
import LoginField from '../LoginField/LoginField.jsx';
import RegisterButton from '../RegisterButton';
import {LoginFormRegisterButton} from '../LoginFormRegisterButton';
import {SignInButton} from '../SignInButton';
import {AuthLoginInputToggle} from '@components/AuthLoginInputToggle';

class LoginForm extends Component {
    constructor(props) {
        super(props);

        this.changeLoginValue = this.changeLoginValue.bind(this);
        this.submitLoginForm = this.submitLoginForm.bind(this);
        this.handlePasswordChange = this.handlePasswordChange.bind(this);
        this.clearFieldError = this.clearFieldError.bind(this);

        this.state = {
            isPasswordChanged: false
        };
    }

    componentDidMount() {
        const {onMount} = this.props;

        if (onMount) {
            onMount();
        }
    }

    componentWillUnmount() {
        this.clearFieldError();
    }

    clearFieldError() {
        const {setLoginError} = this.props;

        setLoginError('');
    }

    changeLoginValue(value) {
        const {updateLoginValue} = this.props;

        updateLoginValue(value);
    }

    handlePasswordChange(event) {
        if (event.target.value) {
            this.setState({isPasswordChanged: true}); // PASSP-22151
        }
    }

    submitLoginForm(event) {
        event.preventDefault();

        const {
            addUserUrl,
            login,
            phoneNumber,
            dispatch,
            isAm,
            isAuthBySWC,
            suggestedLoginFromSWC,
            updatePasswordValue
        } = this.props;

        if (isAm && isAuthBySWC) {
            if (suggestedLoginFromSWC === login) {
                dispatch(
                    sendApiRequest(multiStepAuthStart, {login}, () => {
                        dispatch(sendApiRequest(multiStepAuthCommitPassword, null));
                    })
                );
                return;
            } else {
                dispatch(updatePasswordValue(''));
            }
        }

        dispatch(sendApiRequest(multiStepAuthStart, {login: login || phoneNumber, backPane: addUserUrl}));
    }

    render() {
        const {
            login,
            retpath,
            fretpath,
            clean,
            csrf,
            isCanRegister,
            registrationLogin,
            registrationUrl,
            registrationExpUrl,
            hasCaptcha,
            authUrl,
            loading,
            switchToModeRegisterLite,
            switchToModeRegister,
            registrationType,
            isPreRegisterExp,
            isRegisterWithSuggestToRestoreByPhoneInDaHouse,
            isNeoPhonishRegisterExp,
            isToggleInputExp
        } = this.props;

        return (
            <div
                className={classnames('passp-login-form', {
                    'passp-login-form_can-register': isCanRegister,
                    'passp-login-form_has-captcha': hasCaptcha
                })}
            >
                <form onSubmit={this.submitLoginForm} method='post' action={authUrl}>
                    <input type='hidden' name='retpath' value={retpath} />
                    <input type='hidden' name='fretpath' value={fretpath} />
                    <input type='hidden' name='clean' value={clean} />
                    <input type='hidden' name='service' value={''} />
                    <input type='hidden' name='origin' value={''} />
                    <input type='hidden' name='policy' value={''} />
                    <input type='hidden' name='is_pdd' value={''} />
                    <input type='hidden' name='csrf_token' value={csrf} />

                    {hasCaptcha && <CaptchaField />}

                    {isToggleInputExp ? (
                        <AuthLoginInputToggle
                            login={login}
                            onChangeLogin={this.changeLoginValue}
                            clearFieldError={this.clearFieldError}
                        />
                    ) : (
                        <LoginField login={login} onChangeLogin={this.changeLoginValue} />
                    )}

                    {!this.state.isPasswordChanged && (
                        <input
                            className='passp-hidden-password-field'
                            type='password'
                            id='hiddenPassword'
                            name='hidden-password'
                            onChange={this.handlePasswordChange}
                            tabIndex={-1}
                            autoComplete='off'
                            aria-hidden={true}
                        />
                    )}

                    <RegisterButton
                        login={registrationLogin}
                        url={registrationUrl}
                        isPreRegisterExp={isPreRegisterExp}
                        switchToModeRegisterLite={switchToModeRegisterLite}
                        switchToModeRegister={switchToModeRegister}
                        registrationType={registrationType}
                        isRegisterWithSuggestToRestoreByPhoneInDaHouse={isRegisterWithSuggestToRestoreByPhoneInDaHouse}
                        isNeoPhonishRegisterExp={isNeoPhonishRegisterExp}
                    />

                    <SignInButton isDisabled={loading} />

                    {!isCanRegister && <LoginFormRegisterButton url={registrationExpUrl} />}
                </form>
            </div>
        );
    }
}

LoginForm.defaultProps = {
    retpath: '',
    fretpath: '',
    clean: '',
    csrf: '',
    isCanRegister: false,
    loading: false
};

LoginForm.propTypes = {
    registrationUrl: PropTypes.string.isRequired,
    registrationExpUrl: PropTypes.string.isRequired,
    login: PropTypes.string.isRequired,
    phoneNumber: PropTypes.string,
    retpath: PropTypes.string,
    fretpath: PropTypes.string,
    clean: PropTypes.string,
    csrf: PropTypes.string,
    addUserUrl: PropTypes.string.isRequired,
    authUrl: PropTypes.string.isRequired,
    backPane: PropTypes.string,
    mode: PropTypes.string,
    isCanRegister: PropTypes.bool.isRequired,
    loading: PropTypes.bool.isRequired,
    isAm: PropTypes.bool,
    isAuthBySWC: PropTypes.bool,
    suggestedLoginFromSWC: PropTypes.string,
    registrationLogin: PropTypes.string,
    hasCaptcha: PropTypes.oneOfType([PropTypes.bool, PropTypes.object]),
    updateLoginValue: PropTypes.func.isRequired,
    setupBackPane: PropTypes.func.isRequired,
    canRegister: PropTypes.func.isRequired,
    setLoginError: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    registrationType: PropTypes.string.isRequired,
    switchToModeRegisterLite: PropTypes.func.isRequired,
    switchToModeRegister: PropTypes.func,
    getLiteRegisterTrack: PropTypes.func,
    updatePasswordValue: PropTypes.func,
    isPreRegisterExp: PropTypes.bool,
    isRegisterWithSuggestToRestoreByPhoneInDaHouse: PropTypes.bool,
    isNeoPhonishRegisterExp: PropTypes.bool,
    onMount: PropTypes.func,
    amSetAuthBySWC: PropTypes.func,
    isToggleInputExp: PropTypes.bool
};

export default connect(mapStateToProps, mapDispatchToProps)(LoginForm);
