import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import sendApiRequest from '../../actions/sendApiRequest';
import multiStepAuthCommitPassword from '../../actions/multiStepAuthCommitPassword';
import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import CaptchaField from '../CaptchaField/CaptchaField.jsx';
import CurrentAccount from '../../components/CurrentAccount/CurrentAccount.jsx';
import OTPField from '../OTPField/OTPField.jsx';
import PasswordField from '../PasswordField/PasswordField.jsx';
import {SignInButton} from '../SignInButton';
import {QRButton} from '../QRButton';

const b = cn('AuthPasswordForm');

class PasswordForm extends Component {
    constructor(props) {
        super(props);

        this.changePasswordValue = this.changePasswordValue.bind(this);
        this.submitPasswordForm = this.submitPasswordForm.bind(this);
    }

    componentWillUnmount() {
        const {setPasswordError} = this.props;

        setPasswordError('');
    }

    changePasswordValue(value) {
        const {updatePasswordValue} = this.props;

        updatePasswordValue(value);
    }

    submitPasswordForm(event) {
        event.preventDefault();

        const {password, dispatch} = this.props;

        dispatch(sendApiRequest(multiStepAuthCommitPassword, password));
    }

    render() {
        const {
            password,
            trackId,
            csrf,
            hidePasswordAccountSuggest,
            hasSocialButton,
            hasCaptcha,
            isShowOTPField,
            isShowPasswordField,
            isShowSubmitButton,
            isShouldFallbackToPassword,
            isMagicAuth,
            authUrl,
            loading,
            isShowQrButton
        } = this.props;

        return (
            <div className={b()}>
                <form onSubmit={this.submitPasswordForm} method='post' action={authUrl} noValidate={true}>
                    <input type='hidden' name='track_id' value={trackId} readOnly={true} />
                    <input type='hidden' name='csrf_token' value={csrf} readOnly={true} />

                    {hasCaptcha && (
                        <div className={b('captcha')}>
                            <CaptchaField />
                        </div>
                    )}

                    <div className={b('currentAccount', {hide: hidePasswordAccountSuggest})}>
                        {!hidePasswordAccountSuggest && <CurrentAccount />}
                    </div>

                    {isShowOTPField && <OTPField onChange={this.changePasswordValue} password={password} />}

                    {(isShowPasswordField || isShouldFallbackToPassword) && (
                        <PasswordField
                            onChange={this.changePasswordValue}
                            hasSocialButton={hasSocialButton}
                            password={password}
                        />
                    )}

                    {isShowSubmitButton && (
                        <>
                            <SignInButton isDisabled={loading} />

                            {isMagicAuth && isShowQrButton && (
                                <div className={b('qr-button')}>
                                    <QRButton />
                                </div>
                            )}
                        </>
                    )}
                </form>
            </div>
        );
    }
}

PasswordForm.defaultProps = {
    trackId: '',
    csrf: ''
};

PasswordForm.propTypes = {
    password: PropTypes.string,
    trackId: PropTypes.string,
    csrf: PropTypes.string,
    dispatch: PropTypes.func,
    authUrl: PropTypes.string.isRequired,
    hasSocialButton: PropTypes.bool.isRequired,
    hasCaptcha: PropTypes.oneOfType([PropTypes.bool, PropTypes.object]),
    hidePasswordAccountSuggest: PropTypes.bool.isRequired,
    isShowOTPField: PropTypes.bool.isRequired,
    isShowPasswordField: PropTypes.bool.isRequired,
    isShouldFallbackToPassword: PropTypes.bool.isRequired,
    isMagicAuth: PropTypes.bool.isRequired,
    isShowSubmitButton: PropTypes.bool.isRequired,
    isForceMailAuthEnable: PropTypes.bool.isRequired,
    updatePasswordValue: PropTypes.func.isRequired,
    loading: PropTypes.bool.isRequired,
    setPasswordError: PropTypes.func.isRequired,
    isShowQrButton: PropTypes.bool
};

export default connect(mapStateToProps, mapDispatchToProps)(PasswordForm);
