import React from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';

import {Field} from '@components/Field';
import {Password} from '@components/Password';
import CaptchaField from '../CaptchaField/CaptchaField.jsx';
import {Button} from '@components/Button';
import PhoneConfirmationTimer from '../PhoneConfirmationTimer/PhoneConfirmationTimer.jsx';
import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import errorMessages from '../../errors';
import Title from '../Title/Title.jsx';
import PagePopupContent from '../PagePopupContent/PagePopupContent.jsx';
import Form from '../Form/Form.jsx';
import {CodeField} from '@components/CodeField';

class PhoneConfirmationPopup extends React.PureComponent {
    constructor(props) {
        super(props);

        this.timerInterval = null;
        this.state = {currentTimestamp: Number(new Date())};
    }

    ref = React.createRef();

    componentDidMount() {
        this._stopTimer();
        this._startTimer();
        this.ref && this.ref.current && this.ref.current.focus();
    }

    componentWillUpdate(nextProps) {
        if (this.props.denyResendUntil !== nextProps.denyResendUntil) {
            this.setState({currentTimestamp: Number(new Date())});
            this._startTimer();
        }
    }

    componentWillUnmount() {
        this._handleConfirmationCodeChange('');
        this.props.changeNativeInputValue('confirmation_code', '');
        this.props.setErrors([]);
        this._stopTimer();
    }

    _handleConfirmationCodeChange = (value) => {
        const {updateConfirmationCode} = this.props;

        updateConfirmationCode(value);
    };

    _approvePhone = (event) => {
        const {submitConfirmationCode} = this.props;

        event.preventDefault();
        submitConfirmationCode();
    };

    _sendConfirmationCode = (event) => {
        const {sendConfirmationCode} = this.props;

        event.preventDefault();
        sendConfirmationCode();
    };

    _startTimer = () => {
        this.timerInterval = setInterval(() => {
            const currentTimestamp = Number(new Date());

            this.setState({currentTimestamp});

            if (currentTimestamp >= Number(this.props.denyResendUntil)) {
                this._stopTimer();
            }
        }, 1000);
    };

    _stopTimer = () => {
        if (this.timerInterval) {
            clearInterval(this.timerInterval);
            this.timerInterval = null;
        }
    };

    _getSendAgainButtonText = (showTimer) => {
        const {denyResendUntil} = this.props;
        const {currentTimestamp} = this.state;

        if (showTimer) {
            return (
                <span>
                    {i18n('_AUTH_.enter-confirmation-code.send-again.v2')}
                    {'  '}
                    <PhoneConfirmationTimer startTime={currentTimestamp} endTime={Number(denyResendUntil)} />
                </span>
            );
        }

        return i18n('_AUTH_.enter-confirmation-code.send-again.v2');
    };

    _getPopupTitle = () => {
        const {phoneNumber, securePhoneNumber, amPhoneNumber, isKeyEnabled2FA} = this.props;
        const hasSecondaryAmNumber = amPhoneNumber && amPhoneNumber !== securePhoneNumber;
        const squashedPhoneNumber =
            securePhoneNumber && !hasSecondaryAmNumber
                ? securePhoneNumber.replace(/ /gi, '\u00a0')
                : (phoneNumber || amPhoneNumber).replace(/ /gi, '\u00a0');

        if (isKeyEnabled2FA) {
            return i18n('_AUTH_.phone-confirm.code.title-key-2fa').replace('%phone', squashedPhoneNumber);
        }

        const texts = i18n('_AUTH_.phone-confirm.code.title').split('%phone');

        return (
            <>
                <span dangerouslySetInnerHTML={{__html: texts[0]}} />
                <b>{squashedPhoneNumber}</b>
                {texts.length > 1 && <span dangerouslySetInnerHTML={{__html: texts[1]}} />}
            </>
        );
    };

    render() {
        const {
            confirmationCode,
            isCaptchaRequired,
            denyResendUntil,
            errors,
            isPasswordRequired,
            isKeyEnabled2FA,
            confirmationCodeFieldType,
            hasRoundViewExp
        } = this.props;
        const {currentTimestamp} = this.state;
        const showTimer = currentTimestamp + 1000 < Number(denyResendUntil); // не показывать нулевой таймер

        let fieldError = null;

        if (errors.length) {
            fieldError = errorMessages[errors[0]] ? errorMessages[errors[0]] : errorMessages.internal;
        }

        const sendAgainButtonText = this._getSendAgainButtonText(showTimer);

        return (
            <PagePopupContent cls='phone-confirmation' dataT='phone-confirmation'>
                <Title>{this._getPopupTitle()}</Title>

                <Form cls='phone-confirmation' dataT='phone-confirmation'>
                    {hasRoundViewExp ? (
                        <CodeField codeLength={6} error={fieldError} />
                    ) : (
                        <Field
                            size='l'
                            label={i18n('_AUTH_.phone.enter-confirmation-code.label')}
                            value={confirmationCode}
                            name='confirmation_code'
                            type={confirmationCodeFieldType}
                            options={{inputMode: confirmationCodeFieldType === 'number' ? 'numeric' : 'text'}}
                            error={fieldError}
                            onChange={this._handleConfirmationCodeChange}
                            view='big-input'
                            ref={this.ref}
                        />
                    )}
                    {isPasswordRequired && (
                        <Password view='big-input' label={i18n('_AUTH_.enter_password_title')} skipFocus={true} />
                    )}

                    {isCaptchaRequired && <CaptchaField isCaptchaRequired={true} isShowCaptcha={true} />}

                    <div data-t='phone_enter_confirmation_code_submit' className='passp-button'>
                        <Button
                            type='submit'
                            text={
                                isKeyEnabled2FA
                                    ? i18n('Frontend.next')
                                    : i18n('_AUTH_.phone.enter-confirmation-code.submit')
                            }
                            view='action'
                            size='l'
                            width='max'
                            onClick={this._approvePhone}
                        />
                    </div>

                    <div data-t='phone_send_again' className='passp-button'>
                        <Button
                            type='button'
                            view='pseudo'
                            size='l'
                            width='max'
                            disabled={showTimer}
                            onClick={this._sendConfirmationCode}
                        >
                            {sendAgainButtonText}
                        </Button>
                    </div>
                </Form>
            </PagePopupContent>
        );
    }
}

PhoneConfirmationPopup.propTypes = {
    confirmationCode: PropTypes.string,
    phoneNumber: PropTypes.string,
    amPhoneNumber: PropTypes.string,
    securePhoneNumber: PropTypes.string,
    isCaptchaRequired: PropTypes.bool,
    isKeyEnabled2FA: PropTypes.bool,
    denyResendUntil: PropTypes.number,
    errors: PropTypes.array,
    changeNativeInputValue: PropTypes.func.isRequired,
    updateConfirmationCode: PropTypes.func.isRequired,
    sendConfirmationCode: PropTypes.func.isRequired,
    submitConfirmationCode: PropTypes.func.isRequired,
    setErrors: PropTypes.func.isRequired,
    confirmationCodeFieldType: PropTypes.string.isRequired,
    isPasswordRequired: PropTypes.bool,
    hasRoundViewExp: PropTypes.bool
};

export default connect(mapStateToProps, mapDispatchToProps)(PhoneConfirmationPopup);
