import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import Form from '@blocks/authv2/components/Form/Form.jsx';
import {Field} from '@components/Field';
import {Button} from '@components/Button';
import CaptchaField from '@blocks/authv2/components/CaptchaField/CaptchaField.jsx';
import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import metrics from '@blocks/metrics';
import {PREREGISTER_CLICK_SIMPLE_BTN, PREREGISTER_PAGE} from '@blocks/authv2/metrics_constants';

export class PreRegisterForm extends Component {
    constructor(props) {
        super(props);

        this.state = {
            email: props.login || ''
        };

        this.fieldOptions = {
            autoCorrect: 'off',
            autoCapitalize: 'off',
            autoComplete: 'username'
        };

        this.changeEmail = this.changeEmail.bind(this);
        this.goToRegistration = this.goToRegistration.bind(this);
        this.submit = this.submit.bind(this);
    }

    ref = React.createRef();

    focus = () => {
        if (this.ref.current && this.ref.current.focus) {
            this.ref.current.focus();
        }
    };

    componentDidMount() {
        if (this.props.isNoTabletTouch) {
            return;
        }

        setTimeout(this.focus, 300);
    }

    changeEmail(email) {
        const {isAuthRegisterLiteLoginRequired, updateLoginValue} = this.props;

        if (isAuthRegisterLiteLoginRequired) {
            updateLoginValue(email);
        }
        this.setState({email});
    }

    goToRegistration() {
        const {
            isRegisterWithSuggestToRestoreByPhoneInDaHouse,
            switchToModeRegister,
            isNeoPhonishRegisterExp
        } = this.props;

        metrics.send([PREREGISTER_PAGE, PREREGISTER_CLICK_SIMPLE_BTN]);

        if (isRegisterWithSuggestToRestoreByPhoneInDaHouse || isNeoPhonishRegisterExp) {
            return switchToModeRegister();
        }

        window.location = this.props.registrationUrl;
    }

    submit(event) {
        if (event) {
            event.preventDefault();
        }

        const {isAuthRegisterLiteLoginRequired, goToLiteRegister, multiStepAuthStart} = this.props;

        if (isAuthRegisterLiteLoginRequired) {
            goToLiteRegister();
        } else {
            multiStepAuthStart({login: this.state.email});
        }
    }

    render() {
        const {hasCaptcha, fieldError, isSwitchRegisterExp} = this.props;

        return (
            <Form cls='pre-register' dataT='pre-register'>
                {hasCaptcha && <CaptchaField />}

                <Field
                    size='l'
                    ref={this.ref}
                    label={i18n('_AUTH_.prereg.enter_email.label')}
                    value={this.state.email}
                    name='email'
                    type='text'
                    error={fieldError}
                    options={this.fieldOptions}
                    onChange={this.changeEmail}
                    view='big-input'
                />

                <div data-t='submit_pre_reg' className='passp-button'>
                    <Button
                        type='submit'
                        text={i18n('_AUTH_.next')}
                        view='action'
                        onClick={this.submit}
                        width='max'
                        size='l'
                    />
                </div>

                <div data-t='go_to_reg' className='passp-button'>
                    <Button
                        type='button'
                        text={
                            isSwitchRegisterExp
                                ? i18n('_AUTH_.with_phone_registration')
                                : i18n('_AUTH_.prereg.registration_link')
                        }
                        view='pseudo'
                        width='max'
                        size='l'
                        onClick={this.goToRegistration}
                    />
                </div>
            </Form>
        );
    }
}

PreRegisterForm.propTypes = {
    isNoTabletTouch: PropTypes.bool,
    hasCaptcha: PropTypes.bool,
    isSwitchRegisterExp: PropTypes.bool.isRequired,
    isRegisterWithSuggestToRestoreByPhoneInDaHouse: PropTypes.bool.isRequired,
    isNeoPhonishRegisterExp: PropTypes.bool.isRequired,
    registrationUrl: PropTypes.string.isRequired,
    multiStepAuthStart: PropTypes.func.isRequired,
    fieldError: PropTypes.string,
    isAuthRegisterLiteLoginRequired: PropTypes.bool,
    goToLiteRegister: PropTypes.func.isRequired,
    updateLoginValue: PropTypes.func.isRequired,
    switchToModeRegister: PropTypes.func.isRequired,
    login: PropTypes.string
};

export default connect(mapStateToProps, mapDispatchToProps)(PreRegisterForm);
