import React, {Component} from 'react';
import ReactDOM from 'react-dom';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Button} from '@components/Button';
import {Field} from '@components/Field';
import {changeConfirmationCode, setErrors} from '../../../actions/registerLite';
import sendConfirmationCode from '../../../actions/registerLite/sendConfirmationCode';
import checkConfirmationCode from '../../../actions/registerLite/checkConfirmationCode';
import DomikForm from '@blocks/authv2/components/Form/Form.jsx';
import mapStateToProps from './mapStateToProps';
import {setupBackPane} from '../../../actions/index.js';
import {checkDigitsAndDash} from '@blocks/utils';

class EnterEmailCode extends Component {
    constructor(props) {
        super(props);

        this.fieldOptions = {
            autoCorrect: 'off',
            autoCapitalize: 'off',
            autoComplete: 'off'
        };

        this.checkCode = this.checkCode.bind(this);
        this.updateCodeValue = this.updateCodeValue.bind(this);
        this.sendCodeAgain = this.sendCodeAgain.bind(this);

        this.state = {
            isCodeSentAgain: false
        };

        this.fieldLink = {
            onClick: this.sendCodeAgain,
            pseudo: true,
            text: i18n('_AUTH_.phone_send_code_again')
        };
    }

    componentDidMount() {
        this.props.dispatch(setupBackPane(this.props.addUserUrl));

        setTimeout(() => {
            // eslint-disable-next-line react/no-find-dom-node
            const fieldInput = ReactDOM.findDOMNode(this.confirmationCodeField).querySelector('input');

            fieldInput.value = '';
            fieldInput.focus();
        }, 300);
    }

    checkCode(event) {
        if (event) {
            event.preventDefault();
        }
        this.props.dispatch(checkConfirmationCode());
    }

    updateCodeValue(code) {
        const {dispatch, errorText} = this.props;
        const isValid = checkDigitsAndDash(code);

        if (errorText) {
            dispatch(setErrors({confrimationCode: ''}));
        }

        if (isValid) {
            dispatch(changeConfirmationCode(code));
        }
    }

    sendCodeAgain() {
        this.setState({
            isCodeSentAgain: true
        });
        this.props.dispatch(sendConfirmationCode());
        setTimeout(() => {
            this.setState({
                isCodeSentAgain: false
            });
        }, 2500);
    }

    render() {
        const {confirmationCode, trackId, errorText, errorCode} = this.props;
        const isLimitExceededError = errorCode === 'email.send_limit_exceeded';

        return (
            <DomikForm onSubmit={this.checkCode} method='post' dataT='enter-email-code'>
                <input type='hidden' name='track_id' value={trackId} readOnly={true} />
                <Field
                    size='l'
                    ref={(field) => {
                        this.confirmationCodeField = field;
                    }}
                    name='confirmation-code'
                    type='tel'
                    label={i18n('_AUTH_.code_from_email_label')}
                    value={confirmationCode}
                    error={errorText}
                    options={this.fieldOptions}
                    link={!isLimitExceededError ? this.fieldLink : null}
                    isShowFieldLink={true}
                    onChange={this.updateCodeValue}
                    placeholder='123456'
                    view='big-input'
                />
                {this.state.isCodeSentAgain && !isLimitExceededError && (
                    <div className='passp-lite-form__message'>{i18n('_AUTH_.code_new_send_message')}</div>
                )}
                <div className='passp-button passp-lite-button'>
                    <Button
                        onClick={this.checkCode}
                        type='button'
                        text={i18n('_AUTH_.next')}
                        view='action'
                        size='l'
                        width='max'
                    />
                </div>
            </DomikForm>
        );
    }
}

export default connect(mapStateToProps)(EnterEmailCode);

EnterEmailCode.propTypes = {
    confirmationCode: PropTypes.string,
    addUserUrl: PropTypes.string,
    dispatch: PropTypes.func,
    trackId: PropTypes.string,
    errorText: PropTypes.string,
    errorCode: PropTypes.string
};
