import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import {Field} from '@components/Field';
import {Button} from '@components/Button';
import debounce from 'lodash/debounce';
import DomikForm from '@blocks/authv2/components/Form/Form.jsx';

class Form extends Component {
    constructor(props) {
        super(props);

        this.state = {
            isPasswordShowed: false
        };

        this.submitForm = this.submitForm.bind(this);
        this.handlePassword = this.handlePassword.bind(this);
        this.validatePassword = debounce(this.validatePassword.bind(this), 100);
        this.togglePasswordVision = this.togglePasswordVision.bind(this);
    }

    ref = React.createRef();

    focus = () => {
        if (this.ref.current && this.ref.current.focus) {
            this.ref.current.focus();
        }
    };

    componentDidMount() {
        const {authLiteNameUrl, setupBackPane} = this.props;

        setupBackPane(authLiteNameUrl);

        setTimeout(this.focus, 300);
    }

    togglePasswordVision(event) {
        event.preventDefault();
        event.stopPropagation();

        this.setState((prevState) => ({
            isPasswordShowed: !prevState.isPasswordShowed
        }));
    }

    submitForm(event) {
        const {isPasswordValid, password} = this.props;

        if (event) {
            event.preventDefault();
        }

        if (!password) {
            this.props.validatePassword();
        }

        if (isPasswordValid) {
            this.props.switchToModeRegisterLiteEula();
        }
    }

    handlePassword(password) {
        this.props.changePassword(password);
        this.validatePassword();
    }

    validatePassword() {
        this.props.validatePassword();
    }

    render() {
        const {password, passwordError, passwordWarning, passwordErrorDescription, trackId} = this.props;
        const {isPasswordShowed} = this.state;
        const isPasswordOk = password.length > 6 && !passwordError && !passwordWarning;
        const isPasswordWarningShowed = passwordWarning && !passwordError;

        return (
            <DomikForm onSubmit={this.submitForm} method='post' dataT='register-lite'>
                <div className='passp-register-lite_form'>
                    <input type='hidden' name='track_id' value={trackId} readOnly={true} />
                    <div className='passp-password-field passp-register-lite_password'>
                        <Field
                            size='l'
                            ref={this.ref}
                            name='password'
                            type={isPasswordShowed ? 'text' : 'password'}
                            label={i18n('_AUTH_.field_password')}
                            value={password}
                            error={passwordError}
                            onChange={this.handlePassword}
                            view='big-input'
                        />
                        <span
                            onClick={this.togglePasswordVision}
                            className={classnames('passp-password-field__eye passp-password-field__eye_opened', {
                                'passp-password-field__eye_hidden': isPasswordShowed
                            })}
                        />
                        <span
                            onClick={this.togglePasswordVision}
                            className={classnames('passp-password-field__eye passp-password-field__eye_closed', {
                                'passp-password-field__eye_hidden': !isPasswordShowed
                            })}
                        />
                        {isPasswordOk && (
                            <div className='passp-form-field__password-ok'>{i18n('_AUTH_.password_valid')}</div>
                        )}

                        {isPasswordWarningShowed && <div className='passp-form-field__warning'>{passwordWarning}</div>}
                        <div
                            className={classnames('passp-form-field__error-descr', {
                                'passp-form-field__error-descr-visible': passwordErrorDescription
                            })}
                        >
                            {passwordErrorDescription}
                        </div>
                        <div className='passp-button passp-lite__password-submit'>
                            <Button
                                onClick={this.submitForm}
                                type='button'
                                text={i18n('_AUTH_.next')}
                                view='action'
                                size='l'
                                width='max'
                            />
                        </div>
                    </div>
                </div>
            </DomikForm>
        );
    }
}

export default connect(mapStateToProps, mapDispatchToProps)(Form);

Form.propTypes = {
    trackId: PropTypes.string,
    password: PropTypes.string,
    passwordError: PropTypes.string,
    authLiteRegisterUrl: PropTypes.string,
    passwordWarning: PropTypes.string,
    passwordErrorDescription: PropTypes.string,
    changePassword: PropTypes.func.isRequired,
    validatePassword: PropTypes.func.isRequired,
    setupBackPane: PropTypes.func.isRequired,
    isPasswordValid: PropTypes.bool,
    authLiteNameUrl: PropTypes.string,
    switchToModeRegisterLiteEula: PropTypes.func.isRequired
};
