import React, {Component} from 'react';
import PropTypes from 'prop-types';
import range from 'lodash/range';

const CONFIRMATION_CODE_INPUT_ID = 'confirmation_code_input_';
const CODE_REGEXP = /^.$/;

const errorMessages = {
    internal: i18n('_AUTH_.Errors.internal'),
    undefined: i18n('_AUTH_.errors.required'),
    empty: i18n('_AUTH_.errors.required'),
    notMatched: i18n('_AUTH_.phones.errors.code.invalid'),
    callsLimitExceeded: i18n('_AUTH_.phone-calls-limit-exceeded'),
    smsLimitExceeded: i18n('_AUTH_.phones.errors.limit-exceeded'),
    rateLimitExceeded: i18n('_AUTH_.phones.errors.confirmation-limit-exceeded')
};

class ConfirmationCodeInput extends Component {
    constructor(props) {
        super(props);

        this.handleChange = this.handleChange.bind(this);
        this.getValue = this.getValue.bind(this);
        this.getInputByNum = this.getInputByNum.bind(this);
        this.handleKeyDown = this.handleKeyDown.bind(this);
        this.getInputs = this.getInputs.bind(this);

        this.state = {
            isAutomaticallySend: true,
            confirmationCodeInputs: this.getInputs()
        };
    }

    componentDidMount() {
        const firstInput = this.getInputByNum(1);

        if (firstInput) {
            setTimeout(() => firstInput.focus(), 100);
        }
    }

    componentDidUpdate(prevProps) {
        if (prevProps.confirmationCodeLength !== this.props.confirmationCodeLength) {
            // eslint-disable-next-line react/no-did-update-set-state
            this.setState({confirmationCodeInputs: this.getInputs()});
        }
    }

    getInputs() {
        return range(1, this.props.confirmationCodeLength + 1);
    }

    getInputByNum(num) {
        return document.getElementById(`${CONFIRMATION_CODE_INPUT_ID}${num}`);
    }

    handleKeyDown(event, inputNum) {
        const currentInput = this.getInputByNum(inputNum);

        if (!currentInput) {
            return true;
        }

        if (event.keyCode !== 8) {
            const value = event.target.value;

            if (value && CODE_REGEXP.test(event.key)) {
                currentInput.value = '';
            }

            return true;
        }

        event.preventDefault();

        if (!event.target.value.slice(-1)) {
            if (inputNum !== 1) {
                const previousInput = this.getInputByNum(inputNum - 1);

                if (previousInput) {
                    previousInput.focus();
                }
            }

            return true;
        }

        currentInput.value = '';
        return true;
    }

    getValue() {
        return this.state.confirmationCodeInputs
            .map((inputNum) => {
                const input = this.getInputByNum(inputNum);

                return input ? (input.value || '').trim() : '';
            })
            .join('');
    }

    handleChange(event, inputNum) {
        event.preventDefault();

        const currentInput = this.getInputByNum(inputNum);

        if (!currentInput) {
            return;
        }

        const {confirmationCodeLength, changeCode, sendCode} = this.props;
        const currentInputValue = (currentInput.value || '').trim();

        if (currentInputValue.length > 1) {
            currentInput.value = event.target.value.slice(-1);
        }

        if (inputNum < confirmationCodeLength) {
            const nextInput = this.getInputByNum(inputNum + 1);

            if (nextInput) {
                nextInput.focus();
            }
        }

        const fieldValue = this.getValue();

        changeCode(fieldValue);

        if (fieldValue.length === confirmationCodeLength && this.state.isAutomaticallySend) {
            this.setState({isAutomaticallySend: false});
            sendCode();
        }
    }

    render() {
        const {confirmMethod, codeError} = this.props;
        const fieldError = (codeError && (errorMessages[codeError] || errorMessages.internal)) || '';

        return (
            <div className='passp-confirmation-code-field'>
                <div className='passp-confirmation-code-inputs'>
                    {this.state.confirmationCodeInputs.map((inputNum) => (
                        <input
                            onKeyDown={(event) => this.handleKeyDown(event, inputNum)}
                            onChange={(event) => this.handleChange(event, inputNum)}
                            key={`${confirmMethod}${inputNum}`}
                            type='text'
                            id={`${CONFIRMATION_CODE_INPUT_ID}${inputNum}`}
                            autoComplete='off'
                            autoCapitalize='off'
                            autoCorrect='off'
                        />
                    ))}
                </div>

                {fieldError && (
                    <div className='passp-form-field__error' dangerouslySetInnerHTML={{__html: fieldError}} />
                )}
            </div>
        );
    }
}

ConfirmationCodeInput.propTypes = {
    sendCode: PropTypes.func.isRequired,
    changeCode: PropTypes.func.isRequired,
    codeError: PropTypes.string,
    confirmationCodeLength: PropTypes.number.isRequired,
    confirmMethod: PropTypes.string.isRequired
};

export default ConfirmationCodeInput;
