import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';

import Title from '../../Title/Title.jsx';
import Form from '../../Form/Form.jsx';
import PagePopupContent from '../../PagePopupContent/PagePopupContent.jsx';
import ConfirmationCodeInput from '../ConfirmationCodeInput/ConfirmationCodeInput.jsx';
import Timer from '../Timer/Timer.jsx';
import {Button} from '@components/Button';

import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import metrics from '../../../../metrics';
import {RESTORE_LOGIN_PAGE, RESTORE_LOGIN_SHOW_ENTER_PHONE_CODE_PAGE} from '../../../metrics_constants';
import resizeAuthPopup from '../../../utils/resizeAuthPopup';

class EnterConfirmationCodePopup extends Component {
    constructor(props) {
        super(props);

        this.getSendAgainButtonText = this.getSendAgainButtonText.bind(this);
        this.submitForm = this.submitForm.bind(this);
        this.enableConfirmationCodeResend = this.enableConfirmationCodeResend.bind(this);
    }

    componentDidMount() {
        metrics.send([RESTORE_LOGIN_PAGE, RESTORE_LOGIN_SHOW_ENTER_PHONE_CODE_PAGE]);

        this.props.changeCode('');
        resizeAuthPopup();
    }

    componentDidUpdate(prevProps) {
        if (prevProps.codeError !== this.props.codeError) {
            resizeAuthPopup();
        }
    }

    enableConfirmationCodeResend() {
        this.props.changeCheckPhoneStatus(false);
    }

    getSendAgainButtonText() {
        const {isPhoneCheckSent, sendAgainBtnText} = this.props;

        return (
            <span>
                {sendAgainBtnText}
                {isPhoneCheckSent && <Timer onTimerEnd={this.enableConfirmationCodeResend} />}
            </span>
        );
    }

    submitForm(event) {
        event.preventDefault();
        this.props.confirmPhone();
    }

    render() {
        const {
            confirmPhone,
            checkPhone,
            changeCode,
            codeError,
            title,
            isPhoneCheckSent,
            canSwitchConfirmMethod,
            confirmationCodeLength,
            switchConfirmationMethod,
            isCallConfirmationDisabled,
            confirmMethod
        } = this.props;
        const sendAgainButtonText = this.getSendAgainButtonText();

        return (
            <PagePopupContent cls='restore-login-enter-confirmation-code' dataT='enter-confirmation-code'>
                <Title>
                    <div dangerouslySetInnerHTML={{__html: title}} />
                </Title>

                <Form onSubmit={this.submitForm} cls='enter-confirmation-code' dataT='enter-confirmation-code'>
                    <ConfirmationCodeInput
                        confirmationCodeLength={confirmationCodeLength}
                        confirmMethod={confirmMethod}
                        changeCode={changeCode}
                        sendCode={confirmPhone}
                        codeError={codeError}
                    />

                    <div data-t='submit-phone-code-btn' className='passp-button'>
                        <Button
                            type='submit'
                            text={i18n('_AUTH_.RestoreLogin.next-step')}
                            view='action'
                            size='l'
                            width='max'
                        />
                    </div>

                    <div data-t='send-code-again-btn' className='passp-button passp-send-again-btn'>
                        <Button
                            text={sendAgainButtonText}
                            type='button'
                            view='pseudo'
                            disabled={isPhoneCheckSent || isCallConfirmationDisabled}
                            onClick={checkPhone}
                            size='l'
                            width='max'
                        />
                    </div>

                    {canSwitchConfirmMethod && (
                        <div data-t='switch-confirmation-method-btn' className='passp-button'>
                            <Button
                                text={i18n('_AUTH_.RestoreLogin.confirm_by_sms')}
                                type='button'
                                view='pseudo'
                                disabled={isPhoneCheckSent}
                                onClick={switchConfirmationMethod}
                                size='l'
                                width='max'
                            />
                        </div>
                    )}
                </Form>
            </PagePopupContent>
        );
    }
}

EnterConfirmationCodePopup.propTypes = {
    confirmPhone: PropTypes.func.isRequired,
    changeCheckPhoneStatus: PropTypes.func.isRequired,
    checkPhone: PropTypes.func.isRequired,
    changeCode: PropTypes.func.isRequired,
    switchConfirmationMethod: PropTypes.func.isRequired,
    isPhoneCheckSent: PropTypes.bool.isRequired,
    canSwitchConfirmMethod: PropTypes.bool.isRequired,
    codeError: PropTypes.string.isRequired,
    isCallConfirmationDisabled: PropTypes.bool.isRequired,
    title: PropTypes.string.isRequired,
    sendAgainBtnText: PropTypes.string.isRequired,
    confirmMethod: PropTypes.string.isRequired,
    confirmationCodeLength: PropTypes.number.isRequired
};

export default connect(mapStateToProps, mapDispatchToProps)(EnterConfirmationCodePopup);
