import React, {Component, Fragment} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import mapStateToProps from './mapStateToProps';
import getHintQuestions from '@blocks/common/actions/getHintQuestions';
import {Field} from '@components/Field';
import {Button} from '@components/Button';
import {updateError, updateHintData} from '@blocks/authv2/actions/restorePassword';
import validateHintData from '@blocks/authv2/actions/restorePassword/validateHintData';
import goToNextStep from '@blocks/authv2/actions/restorePassword/goToNextStep';
import UserLogin from '@blocks/authv2/components/RestorePassword/UserLogin/UserLogin';
import {Select} from '@components/Select';

class BindHint extends Component {
    componentDidMount() {
        this.props.dispatch(getHintQuestions());
    }

    handleChange = (value) => this.props.dispatch(updateHintData({answer: value}));

    handleCustomQuestion = (value) => this.props.dispatch(updateHintData({customQuestion: value}));

    handleSubmit = (e) => {
        e.preventDefault();

        const {bindHint, dispatch} = this.props;
        const {questionId = '', questionText = '', answer = ''} = bindHint;
        const params = {
            hint_question_id: questionId,
            hint_answer: answer
        };

        dispatch(updateHintData({errorField: ''}));

        if (!questionId.trim() || questionId === '0') {
            dispatch(updateHintData({errorField: 'question_id'}));
            dispatch(
                updateError({
                    code: 'missingvalue',
                    errorText: i18n('_AUTH_.hint_question_id_errors_missingvalue')
                })
            );
            return;
        }

        if (!answer.trim()) {
            dispatch(updateHintData({errorField: 'hint_answer'}));
            dispatch(
                updateError({
                    code: 'missingvalue',
                    errorText: i18n('_AUTH_.hint_answer_errors_missingvalue')
                })
            );
            return;
        }

        if (questionId === '99') {
            params.hint_question = questionText;

            if (!questionText.trim()) {
                dispatch(updateHintData({errorField: 'hint_question'}));
                dispatch(
                    updateError({
                        code: 'missingvalue',
                        errorText: i18n('_AUTH_.hint_question_id_errors_missingvalue')
                    })
                );
                return;
            }
        }

        this.props.dispatch(validateHintData(params));
    };

    onSelectChange = (event) => {
        const selectVal = event.target.value;
        const {bindHint, questionsList, dispatch} = this.props;
        const selected = questionsList.find((question) => question.val === selectVal);
        const questionData = {
            questionId: selected.val,
            questionText: selected.text
        };

        if (bindHint.errorField === 'question_id') {
            dispatch(updateHintData({errorField: ''}));
        }
        dispatch(updateHintData(questionData));
    };

    toggleBindMethod = () => {
        this.props.dispatch(goToNextStep('bind-phone'));
    };

    render() {
        const {trackId, questionsList, bindHint, errorText} = this.props;

        return (
            <Fragment>
                <UserLogin />
                <form onSubmit={this.handleSubmit} action='/auth/restore/password/bind-hint'>
                    <input type='hidden' name='track_id' value={trackId} readOnly={true} />
                    <label htmlFor='control_question' className='question-select__label'>
                        {i18n('_AUTH_.field_hint_question_id')}
                    </label>

                    <div className='restore-hint-select' data-t='user-question-all'>
                        <Select
                            size='l'
                            onChange={this.onSelectChange}
                            onBlur={this.onSelectChange}
                            name='hint_question_id'
                            id='hint_question_id'
                            width='max'
                            dataT='hint_question_id'
                            value={bindHint.questionId || (questionsList[0] && questionsList[0].val)}
                            options={questionsList.map((question) => ({value: question.val, content: question.text}))}
                        />
                        {bindHint.errorField === 'question_id' && (
                            <div className='passp-form-field__error' aria-live='polite'>
                                {errorText}
                            </div>
                        )}
                    </div>

                    {bindHint.questionId === '99' && (
                        <div>
                            <Field
                                size='l'
                                label={i18n('_AUTH_.question_enter_custom')}
                                value={bindHint.customQuestion || ''}
                                name='hint_custom-question'
                                type='text'
                                error={bindHint.errorField === 'hint_custom-question' ? errorText : ''}
                                onChange={this.handleCustomQuestion}
                                view='big-input'
                            />
                        </div>
                    )}

                    <Field
                        size='l'
                        label={i18n('_AUTH_.field_hint_answer')}
                        value={bindHint.answer || ''}
                        name='hint_answer'
                        type='text'
                        error={bindHint.errorField === 'hint_answer' ? errorText : ''}
                        onChange={this.handleChange}
                        view='big-input'
                    />

                    <div data-t='restore-bind-question' className='passp-button'>
                        <Button
                            type='submit'
                            onClick={this.handleSubmit}
                            text={i18n('_AUTH_.next_registration_step')}
                            view='action'
                            size='l'
                            width='max'
                        />
                    </div>

                    <div data-t='restore-toggle-phone' className='passp-button'>
                        <Button
                            type='button'
                            onClick={this.toggleBindMethod}
                            text={i18n('_AUTH_.phone_number_registration')}
                            view='pseudo'
                            size='l'
                            width='max'
                        />
                    </div>
                </form>
            </Fragment>
        );
    }
}

export default connect(mapStateToProps)(BindHint);

BindHint.propTypes = {
    dispatch: PropTypes.func.isRequired,
    bindHint: PropTypes.shape({
        questionId: PropTypes.string.isRequired,
        questionText: PropTypes.string.isRequired,
        answer: PropTypes.string.isRequired,
        errorField: PropTypes.string,
        customQuestion: PropTypes.string
    }),
    questionsList: PropTypes.array,
    trackId: PropTypes.string,
    errorText: PropTypes.string
};
