import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';

import {classNames} from '../../../../utils';
import mapStateToProps from './mapStateToProps';

import {Field} from '@components/Field';
import {Button} from '@components/Button';

import processError from '../../../actions/restorePassword/processError';
import goToNextStep from '../../../actions/restorePassword/goToNextStep';
import getStartData from '../../../actions/restorePassword/getStartData';
import switchToModeRestoreLogin from '@blocks/authv2/actions/switchToModeRestoreLogin';
import {setupMode} from '@blocks/authv2/actions';
import {setupBackPane, updateLoginValue} from '../../../actions';
import metrics from '../../../../metrics';

import LoginSuggest from './LoginsSuggest.jsx';
import {updateError} from '../../../actions/restorePassword';
import {RESTORE_PASSWORD, RESTORE_PASSWORD_SHOW_LOGIN, RESTORE_PASSWORD_SELECT_LOGIN} from '../../../metrics_constants';

const defaultCssClass = classNames('auth-restore-pass');

class EnterLogin extends Component {
    constructor(props) {
        super(props);
        this.loginField = React.createRef();

        const {fieldLink, dispatch} = props;

        this.fieldLink = Object.assign({}, fieldLink);
        this.fieldLink.onClick = () => {
            dispatch(switchToModeRestoreLogin());
        };
    }

    componentDidMount() {
        const {dispatch, addUserUrl, isAuthDataAvailable} = this.props;

        this.loginField.current.focus();

        dispatch(
            updateError({
                code: '',
                errorText: ''
            })
        );

        this.getData();

        if (isAuthDataAvailable) {
            dispatch(setupBackPane(addUserUrl));
        }

        dispatch(setupMode('restore-password'));
        metrics.send([RESTORE_PASSWORD, RESTORE_PASSWORD_SHOW_LOGIN]);
    }

    getData = () => {
        const {dispatch} = this.props;

        dispatch(getStartData());
    };

    handleSubmit = (event) => {
        const {login, dispatch} = this.props;

        event.preventDefault();

        if (!login.trim()) {
            dispatch(processError('login.empty'));
            return;
        }

        dispatch(goToNextStep('captcha'));
    };

    handleChange = (val) => {
        const {dispatch} = this.props;

        dispatch(updateLoginValue(val));
    };

    selectLogin = (event) => {
        const {dispatch} = this.props;

        if (!event || !event.target) {
            return;
        }

        dispatch(updateLoginValue(event.target.value));
        metrics.send([RESTORE_PASSWORD, RESTORE_PASSWORD_SELECT_LOGIN]);
    };

    render() {
        const {logins, login, errorText, trackId} = this.props;

        return (
            <form
                className={`${defaultCssClass('form')} passp-login-form`}
                onSubmit={this.handleSubmit}
                action='/restore/password'
            >
                <input type='hidden' name='track_id' value={trackId} readOnly={true} />
                <Field
                    size='l'
                    ref={this.loginField}
                    label={i18n('_AUTH_.login_id.label')}
                    value={login}
                    name='login'
                    type='text'
                    link={this.fieldLink}
                    isShowFieldLink={true}
                    error={errorText}
                    onChange={this.handleChange}
                    placeholder={i18n('_AUTH_.login_id_placeholder')}
                    view='big-input'
                />
                <LoginSuggest selected={login} onChange={this.selectLogin} logins={logins} />
                <div data-t='restore-pwd_sumbit' className='passp-button'>
                    <Button
                        onClick={this.handleSubmit}
                        type='submit'
                        text={i18n('_AUTH_.next_registration_step')}
                        view='action'
                        size='l'
                        width='max'
                    />
                </div>
            </form>
        );
    }
}

export default connect(mapStateToProps)(EnterLogin);

EnterLogin.propTypes = {
    logins: PropTypes.array,
    login: PropTypes.string,
    errorText: PropTypes.string,
    trackId: PropTypes.string,
    addUserUrl: PropTypes.string,
    fieldLink: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    isAuthDataAvailable: PropTypes.bool
};
