import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {SHOW_LOGIN_FORM, START_SOCIAL_AUTH} from '@blocks/authv2/metrics_constants';
import metrics from '@blocks/metrics';
import {Icon} from '@components/Icon';
import {Popup} from '@components/Popup';

const b = cn('AuthSocialBlock');

export class SocialBlock extends Component {
    secondaryButtonRef = React.createRef();

    constructor(props) {
        super(props);

        this.state = {
            showSecondaryProviders: false
        };
    }

    authSocial = (data) => {
        const {switchToModeMagic, startSocialAuth, socialScope} = this.props;
        const {code, name} = data;

        if (code === 'yandex') {
            switchToModeMagic(true);
            return;
        }

        if (code === 'autoRu') {
            this.sendAutoRuClickMetricks();
        } else {
            metrics.send([SHOW_LOGIN_FORM, START_SOCIAL_AUTH, name || code]);
        }

        startSocialAuth({
            provider: code,
            scope: socialScope && socialScope[code]
        });
    };

    sendAutoRuClickMetricks = () => {
        metrics.send([SHOW_LOGIN_FORM, 'AutoRuButtonClick']);
    };

    auth = (event, data = {}) => {
        const {switchToModeMagic} = this.props;
        const {code} = data;

        event.preventDefault();

        this.hideSecondaryProviders();

        if (['qr', 'yandex'].includes(code)) {
            return switchToModeMagic(code === 'yandex');
        }

        this.authSocial(data);
    };

    showSecondaryProviders = () => this.setState({showSecondaryProviders: true});
    hideSecondaryProviders = () => this.setState({showSecondaryProviders: false});

    renderButtonContent = ({data = {}, type = 'primary'} = {}) => {
        const {language, socialButtonsWithText} = this.props;
        const {code, display_name: displayName} = data;
        const name = displayName[language] || displayName.default;

        if (['autoRu', 'qr'].includes(code)) {
            return (
                <>
                    <span className={b('icon', {withHover: false})} />
                    <span className={b('name', {withHover: false})}>{name}</span>
                </>
            );
        }

        if (socialButtonsWithText) {
            return (
                <>
                    <Icon size='s' type={data.code} withHover={false} />
                    <span className={b('name', {withHover: false})}>{name}</span>
                </>
            );
        }

        if (type === 'secondary') {
            return (
                <>
                    <Icon size='s' type={code} />
                    <span className={b('secondaryProviderName')}>{name}</span>
                </>
            );
        }

        if (type === 'primary') {
            return (
                <>
                    <Icon size='s' type={data.code} withHover={false} />
                </>
            );
        }
    };

    renderProviderButton = ({id, data, type = 'primary', hidden = false} = {}) => {
        const {socialButtonsWithText, isAm} = this.props;
        const {display_name: displayName = {}, code = '', link = ''} = data;

        if (isAm && id === 'autoRu') {
            return (
                <a
                    key={id}
                    href={link}
                    onClick={() => this.sendAutoRuClickMetricks()}
                    className={b('provider', {code, withText: socialButtonsWithText})}
                    data-t={`provider:primary:${code}`}
                    aria-label={displayName.default}
                >
                    {this.renderButtonContent({data, type})}
                </a>
            );
        }

        if (type === 'primary') {
            return (
                <button
                    key={id}
                    type='button'
                    onClick={(event) => this.auth(event, data)}
                    className={b('provider', {code: data.code, withText: socialButtonsWithText})}
                    data-t={`provider:primary:${data.code}`}
                    aria-label={displayName.default}
                >
                    {this.renderButtonContent({data, type})}
                </button>
            );
        }

        if (type === 'secondary') {
            return (
                <button
                    key={id}
                    type='button'
                    onClick={(event) => this.auth(event, data)}
                    className={b('secondaryProvider', {hidden})}
                    data-t={`provider:primary:${code}`}
                    aria-label={displayName.default}
                >
                    {this.renderButtonContent({data, type})}
                </button>
            );
        }
    };

    renderProviders = ({type = 'primary'} = {}) => {
        const {showSecondaryProviders} = this.state;
        const {providers: {primary = [], secondary = []} = {}, dir} = this.props;

        switch (type) {
            case 'primary':
                return primary.map(({id, data}) => this.renderProviderButton({id, data, type}));
            case 'secondary':
                return (
                    <>
                        {secondary.map(({id, data = {}}) => this.renderProviderButton({id, data, type, hidden: true}))}
                        <button
                            type='button'
                            className={b('provider')}
                            ref={this.secondaryButtonRef}
                            onClick={this.showSecondaryProviders}
                            data-t='provider:more'
                            aria-hidden='true'
                            tabIndex='-1'
                        >
                            <Icon size='s' type={'dots'} withHover={true} />
                        </button>
                        <Popup
                            target='anchor'
                            dataT='provider:secondary:popup'
                            visible={showSecondaryProviders}
                            anchor={this.secondaryButtonRef}
                            directions={dir === 'rtl' ? ['top-center', 'right-center'] : ['top-center', 'left-center']}
                            onClose={this.hideSecondaryProviders}
                        >
                            <div className={b('secondaryProviders')}>
                                {secondary.map(({id, data = {}}) => this.renderProviderButton({id, data, type}))}
                            </div>
                        </Popup>
                    </>
                );
        }
    };

    render() {
        const {providers = {}, isRegister, minimal, intranet} = this.props;
        const {primary = [], secondary = []} = providers;

        if (minimal || intranet || (!primary.length && !secondary.length)) {
            return null;
        }

        const title = isRegister ? i18n('_AUTH_.prereg.social_title') : i18n('_AUTH_.AuthQR.promo.social_title');

        return (
            <div className={b()}>
                <h2 className={b('title')}>{title}</h2>
                <div className={b('providers')}>
                    {this.renderProviders({type: 'primary'})}
                    {Boolean(secondary.length) && this.renderProviders({type: 'secondary'})}
                </div>
            </div>
        );
    }
}

SocialBlock.defaultProps = {
    minimal: false,
    intranet: false,
    isRegister: false
};

SocialBlock.propTypes = {
    mode: PropTypes.string,
    minimal: PropTypes.bool.isRequired,
    intranet: PropTypes.bool.isRequired,
    isAm: PropTypes.bool,
    socialButtonsWithText: PropTypes.bool.isRequired,
    socialScope: PropTypes.object,
    switchToModeMagic: PropTypes.func.isRequired,
    startSocialAuth: PropTypes.func.isRequired,
    isRegister: PropTypes.bool,
    language: PropTypes.string.isRequired,
    providers: PropTypes.shape({
        primary: PropTypes.array,
        secondary: PropTypes.array
    }),
    env: PropTypes.string,
    dir: PropTypes.string
};
