import {setMagicError, changeCaptchaState, redirect} from './actions';
import {showMagicSuccess} from './actions/showMagicSuccess';
import {setCaptchaTrack} from '@components/Captcha/actions';
import {hasExp} from '@blocks/selectors';

import showAccounts from './actions/showAccounts';
import reAuthPasswordSubmit from './actions/reAuthPasswordSubmit';
import {SUCCESS_AUTH_GOAL} from './metrics_constants';
import metrics from '@blocks/metrics';

const QR_SUCCESS_ANIMATION_DURATION = 2500; // milliseconds

let store;
const magicWrapper = {
    init(s) {
        this.setStore(s);
        try {
            this._isSliderConfirmationDesignExp = this.getStore()
                .getState()
                .common.experiments.flags.includes('slider-confirmation-design');
        } catch (error) {
            this._isSliderConfirmationDesignExp = false;
        }
    },

    onSuccess(message) {
        let state;

        if (message.origin !== location.origin) {
            return;
        }

        if (message.data.socialAuth) {
            const {getState, dispatch} = this.getStore();

            state = getState();

            if (state.common.retpath) {
                redirect(state.common.retpath);
            } else {
                dispatch(showAccounts());
            }
        }
    },

    setStore(s) {
        store = s;
    },

    getStore() {
        return store;
    },

    start(params) {
        this.restart(params);
    },

    restart() {
        const self = this;
        const state = this.getStore().getState();

        self.track_id = state.auth.magicTrack;
        self.csrf_token = state.auth.magicCSRF;

        self.trackNotFoundErrorCount = 0;
        self.timeoutErrorCount = 0;

        self.stop();

        self._interval = 300;

        window.intervalTimeout = setTimeout(function() {
            self._interval = 1000;
        }, 60 * 1000);

        window.redirectTimeout = setTimeout(function() {
            window.location.reload(true);
        }, 60 * 1000 * 10);

        self._stopped = false;
        self.restartPolling();
    },

    restartPolling() {
        const self = this;

        if (!self._stopped) {
            setTimeout(function() {
                self.poll();
            }, self._interval);
        }
    },

    poll() {
        const self = this;
        const state = this.getStore().getState();
        const {processedAccount: account, typeImage2FA} = state.auth;
        const authMethods = (account && account.allowed_auth_methods) || [];
        const isAccountWith2FA = authMethods.includes('magic') && authMethods.includes('otp');
        const is2faWithImage = isAccountWith2FA && hasExp(state, 'is-2fa-with-image');
        const handler =
            is2faWithImage && typeImage2FA === 'image'
                ? '/registration-validations/auth/multi_step/commit_magic'
                : '/auth/new/magic/status/';
        const req = $.ajax(handler, {
            data: {
                track_id: self.track_id,
                csrf_token: self.csrf_token
            },
            dataType: 'json',
            timeout: 10000,
            type: 'POST'
        });

        const {dispatch} = self.getStore();

        req.fail(function() {
            if (self.timeoutErrorCount > 1) {
                self.stop();
                dispatch(setMagicError('global'));
            }

            self.timeoutErrorCount++;
            self.restartPolling();
        });

        req.done(function(results) {
            let error;

            if (results.state === 'auth_challenge') {
                self.stop();
                window.location.href = results.redirectUrl;
                return;
            }

            if (results.status === 'ok' && results.state === 'otp_auth_finished') {
                self.stop();
                metrics.goal(SUCCESS_AUTH_GOAL);

                const finishUrl = `/auth/finish/?track_id=${self.track_id}`;

                if (self._isSliderConfirmationDesignExp) {
                    dispatch(showMagicSuccess());
                    setTimeout(() => {
                        window.location.href = finishUrl;
                    }, QR_SUCCESS_ANIMATION_DURATION);
                } else {
                    window.location.href = finishUrl;
                }
                return;
            }

            if (results.errors) {
                error = results.errors[0];

                if (error === 'password.not_matched') {
                    error = 'password.not_matched_2fa';
                }

                if (error === 'captcha.required') {
                    self.stop();
                    dispatch(setCaptchaTrack(self.track_id));
                    dispatch(changeCaptchaState(true));
                    return;
                }

                if (error === 'track.not_found') {
                    // TODO: убрать костыль для PASSP-21000
                    if (self.trackNotFoundErrorCount > 5) {
                        self.stop();
                        dispatch(setMagicError(error));
                        dispatch(reAuthPasswordSubmit());
                    }

                    self.trackNotFoundErrorCount++;
                }

                if (['internal.temporary', 'backend.redis_failed', 'track.not_found'].indexOf(error) === -1) {
                    self.stop();
                    dispatch(setMagicError(error));
                    dispatch(reAuthPasswordSubmit());
                }
            }

            self.restartPolling();
        });
    },

    clearCaptchaState() {
        const {dispatch} = this.getStore();

        dispatch(changeCaptchaState(false));
    },

    stopPoll() {
        this._stopped = true;
    },

    stop() {
        this.stopPoll();

        clearTimeout(window.intervalTimeout);
        clearTimeout(window.redirectTimeout);
    }
};

export default magicWrapper;
