import {v4 as uuid} from 'uuid';
import metrika from '@blocks/metrics';
import {Platforms} from './constants';

export class NativeMobileApiTransport {
    _outgoingRequests = {};
    _eventsLog = [];
    _onceMessagesSend = {};
    _eventsSubscriber;

    constructor(version, platform, options = {}) {
        this.version = version;
        this.platform = platform;
        this.options = options;

        this._nativeSendMessage = this._getNativeMessageSender();

        window.nativeAMResponse = {
            receive: this._getNativeMessageReceiver()
        };
    }

    sendMessageToNativeApi(message, data, callback) {
        const requestId = uuid();

        const payload = {
            version: this.version,
            message,
            requestId,
            data
        };

        if (callback && typeof callback === 'function') {
            this._outgoingRequests[requestId] = callback;
        }

        this._nativeSendMessage(payload);
    }

    sendMessageToNativeApiOnce(message, data, callback) {
        if (this._onceMessagesSend[message]) {
            return;
        }

        this._onceMessagesSend[message] = true;

        this.sendMessageToNativeApi(message, data, callback);
    }

    setSubscriber(subscriber) {
        this._eventsSubscriber = subscriber;

        return function unsubscribe() {
            this._eventsSubscriber = undefined;
        }.bind(this);
    }

    getEvents() {
        return this._eventsLog;
    }

    _getNativeMessageReceiver() {
        return (requestId, payload) => {
            payload = this._normalizePayload(payload);
            const receiver = this._outgoingRequests[requestId];

            if (receiver && typeof receiver === 'function') {
                delete this._outgoingRequests[requestId];

                this._safeCallReceiver(receiver, payload);
            }

            this._logEvent('incoming', {requestId, payload});
        };
    }

    _getNativeMessageSender() {
        let messageSender = () => {
            metrika.send(`Попытка отправить сообщение нативному слою на неизвестной платформе ${this.platform}`);
        };

        if (this.platform === Platforms.ANDROID && window.nativeAMAndroid && window.nativeAMAndroid.send) {
            messageSender = this._createMessageSender((payload) =>
                window.nativeAMAndroid.send(JSON.stringify(payload))
            );
        } else if (
            this.platform === Platforms.IOS &&
            window.webkit &&
            window.webkit.messageHandlers &&
            window.webkit.messageHandlers.nativeAM &&
            window.webkit.messageHandlers.nativeAM.postMessage
        ) {
            messageSender = this._createMessageSender((payload) =>
                window.webkit.messageHandlers.nativeAM.postMessage(payload)
            );
        }

        return messageSender;
    }

    _createMessageSender(sendMessage) {
        return (message) => {
            sendMessage(message);

            this._logEvent('outgoing', message);
        };
    }

    _normalizePayload(payload = {}) {
        try {
            if (typeof payload === 'string') {
                return JSON.parse(payload);
            }

            return payload;
        } catch (error) {
            return payload;
        }
    }

    _safeCallReceiver(receiver, payload) {
        try {
            receiver(payload);
        } catch (error) {
            metrika.send(`[AM] Ошибка при обработке нативного сообщения`);
        }
    }

    _logEvent(eventDirection, message) {
        if (!this.options.logEvents) {
            return;
        }

        const event = {
            direction: eventDirection,
            message,
            date: new Date()
        };

        this._eventsLog.push(event);

        if (this._eventsSubscriber && typeof this._eventsSubscriber === 'function') {
            this._eventsSubscriber(event);
        }
    }
}
