import React from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {useDispatch} from 'react-redux';
import {goBack, push} from 'react-router-redux';
import {Field} from '@components/Field';
import {NATIVE_MOBILE_API_OUTGOING_ACTIONS} from '@blocks/authv2/actions/nativeMobileApi';
import {Select} from '@components/Select';

import './AmDebug.styl';

const nativeMobileApiTransport = (() => {
    if (typeof window === 'object') {
        return window.__AM_NATIVE_MOBILE_TRANSPORT__;
    }
})();

const outgoingActions = Object.keys(NATIVE_MOBILE_API_OUTGOING_ACTIONS).map((key) => ({
    value: key,
    content: NATIVE_MOBILE_API_OUTGOING_ACTIONS[key]
}));

const actionFields = {
    AUTH_SUCCEED: ['login', 'password', 'avatarUrl'],
    READY: ['status'],
    START_SOCIAL_AUTH: ['provider'],
    SEND_METRICS: ['identifier', 'parameters']
};

export function AmDebug({finishOkUrl, finishErrorUrl}) {
    const dispatch = useDispatch();
    const [events, setEvents] = React.useState([]);
    const [eventType, setEventType] = React.useState(outgoingActions[0].value);
    const [eventPayload, setEventPayload] = React.useState({});
    const [isThrowError, setIsThrowError] = React.useState(false);

    React.useEffect(() => {
        if (isThrowError) {
            throw new Error('everybodybecoolthisisdebug');
        }
    }, [isThrowError]);

    React.useEffect(() => {
        if (!nativeMobileApiTransport) {
            return;
        }

        setEvents(nativeMobileApiTransport.getEvents());

        const unsubscribe = nativeMobileApiTransport.setSubscriber(() => {
            setEvents(nativeMobileApiTransport.getEvents());
        });

        return () => {
            unsubscribe();
        };
    }, []);

    const createEventPayloadHandler = (payloadField) => (event) => {
        setEventPayload((currPayload) => ({
            ...currPayload,
            [payloadField]: event
        }));
    };

    const handleOkFinish = React.useCallback(() => {
        dispatch(push(finishOkUrl));
    }, [finishOkUrl, dispatch]);

    const handleErrorFinish = React.useCallback(() => {
        dispatch(push(finishErrorUrl));
    }, [finishErrorUrl, dispatch]);

    const handleBack = React.useCallback(() => {
        dispatch(goBack());
    }, [dispatch]);

    const handleThrowUncatchedException = React.useCallback(() => {
        setIsThrowError(true);
    }, []);

    const handleEventTypeChange = React.useCallback((event) => {
        setEventType(event.target.value);
    }, []);

    const handleSendEvent = React.useCallback(() => {
        dispatch({
            type: NATIVE_MOBILE_API_OUTGOING_ACTIONS[eventType],
            payload: eventPayload
        });
    }, [eventType, eventPayload, dispatch]);

    return (
        <div>
            <div className='am-debug__controls'>
                <div>
                    <Button text='Назад' onClick={handleBack} size='m' />
                </div>
                <div>
                    <Button text='finish(ok)' onClick={handleOkFinish} size='m' className='am-debug__controls_button' />
                </div>
                <div>
                    <Button
                        text='finish(error)'
                        onClick={handleErrorFinish}
                        size='m'
                        className='am-debug__controls_button'
                    />
                </div>
            </div>
            <div className='am-debug__controls'>
                <div>
                    <Button
                        text='Критическая ошибка'
                        onClick={handleThrowUncatchedException}
                        size='m'
                        className='am-debug__controls_button'
                    />
                </div>
            </div>
            <div>
                <p className='am-debug_title'>Отправить событие</p>
                <Select
                    size='l'
                    onChange={handleEventTypeChange}
                    name='event_type'
                    id='event_type'
                    width='max'
                    value={eventType}
                    options={outgoingActions}
                    className='am-debug_event-type'
                />
                {actionFields[eventType] &&
                    actionFields[eventType].map((payloadType) => (
                        <Field
                            key={payloadType}
                            name={`field-${payloadType}`}
                            label={payloadType}
                            view='floating-label'
                            size='l'
                            options={{
                                autoComplete: 'off',
                                autoCapitalize: 'off',
                                autoCorrect: 'off'
                            }}
                            value={eventPayload[payloadType]}
                            onChange={createEventPayloadHandler(payloadType)}
                        />
                    ))}
                <Button text='Отправить' onClick={handleSendEvent} size='m' className='am-debug__controls_button' />
            </div>
            <div className='am-debug__events-log'>
                <p className='am-debug_title'>Лог событий:</p>
                <ul>
                    {events.map((event) => (
                        <li key={event.message.requestId} className='am-debug__events-log_event'>
                            <p>{event.direction === 'incoming' ? '➡️ am->web' : '⬅️ web->am'}</p>
                            <span className='am-debug__events-log_event_date'>{event.date.toISOString()}</span>
                            <pre className='am-debug__events-log_event_message'>
                                {JSON.stringify(event.message, null, 2)}
                            </pre>
                        </li>
                    ))}
                </ul>
            </div>
        </div>
    );
}

AmDebug.propTypes = {
    finishOkUrl: PropTypes.string.isRequired,
    finishErrorUrl: PropTypes.string.isRequired
};
