import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import Page from '@blocks/authv2/components/Page/Page.jsx';
import Title from '@blocks/authv2/components/Title/Title.jsx';
import {DomikFooterLink} from '@blocks/authv2/components/DomikFooterLink';
import ChallengeWithEmail from '@blocks/authv2/components/Challenges/ChallengeWithEmail/ChallengeWithEmail.jsx';
import ChallengeWithPhone from '@blocks/authv2/components/Challenges/ChallengeWithPhone/ChallengeWithPhone.jsx';
import ChallengeWithPush from '@blocks/authv2/components/Challenges/ChallengeWithPush/ChallengeWithPush.jsx';
import ChallengeWithQuestion from '@blocks/authv2/components/Challenges/ChallengeWithQuestion/ChallengeWithQuestion';
import ChallengeWith3ds from '@blocks/authv2/components/Challenges/ChallengeWith3ds/ChallengeWith3ds';
import ChallengeWithEmailCode from '@blocks/authv2/components/Challenges/ChallengeWithEmailCode/ChallengeWithEmailCode';
// eslint-disable-next-line max-len
import ChallengeWithPhoneConfirmation from '@blocks/authv2/components/Challenges/ChallengeWithPhoneConfirmation/ChallengeWithPhoneConfirmation.jsx';
import ChallengeWithDictation from '../../components/Challenges/ChallengeWithDictation/ChallengeWithDictation';
import {updateFieldValue, setChallengeError} from '@blocks/authv2/actions/challenge';
import commitChallenge from '@blocks/authv2/actions/challenge/commitChallenge';
import {commit3ds} from '@blocks/UserValidate/actions/commit3ds';
import {submit3ds} from '@blocks/UserValidate/actions/submit3ds';
import {set3dsLoadingStatus, setTrustFrameOpenState} from '@blocks/UserValidate/actions/';
import {redirectToErrorFrom3ds} from '@blocks/UserValidate/actions/redirectToErrorFrom3ds';
import sendPush from '@blocks/authv2/actions/challenge/sendPush';
import sendCodeToEmail from '@blocks/authv2/actions/challenge/sendCodeToEmail';
import confirmEmailCode from '@blocks/authv2/actions/challenge/confirmEmailCode';
import alternateChallengeType from '@blocks/authv2/actions/challenge/alternateChallengeType';
import validatePhone from '@blocks/authv2/actions/challenge/validatePhone';
import switchToRestorePassword from '@blocks/authv2/actions/switchToRestorePassword';
import metrics from '@blocks/metrics';
import {classNames, getFirstSymbolForQuery} from '@blocks/utils';
import {redirect} from '@blocks/authv2/actions';
import mapStateToProps from './mapStatesToProps';
import {
    CHALLENGES_PAGE,
    CHALLENGES_PAGE_PHONE_RESTORE,
    CHALLENGES_PAGE_EMAIL_RESTORE,
    CHALLENGES_PAGE_PHONE_CALL_RESTORE,
    CHALLENGES_PAGE_EMAIL_CODE_RESTORE
} from '@blocks/authv2/metrics_constants';
import {formatLocTextWithLink} from '@blocks/authv2/utils/formatLocTextWithLink';
import {Button} from '@plibs/pcomponents/Button';

const defaultCssClass = classNames('auth-challenge');
const supportFormLink = '/restoration/form?unconditional=true';

class Challenge extends Component {
    updateFieldValue = (value) => {
        this.props.dispatch(updateFieldValue(value));
    };

    passChallenge = () => {
        this.props.dispatch(commitChallenge());
    };

    validatePhone = () => {
        this.props.dispatch(validatePhone());
    };

    toggleChallenge = () => {
        this.props.dispatch(alternateChallengeType());
    };

    sendPush = () => {
        this.props.dispatch(sendPush());
    };

    sendCodeToEmail = () => {
        this.props.dispatch(sendCodeToEmail());
    };

    confirmEmailCode = () => {
        this.props.dispatch(confirmEmailCode());
    };

    commit3ds = () => {
        this.props.dispatch(commit3ds());
    };

    submit3ds = () => {
        this.props.dispatch(submit3ds());
    };

    set3dsLoadingStatus = (bool) => {
        this.props.dispatch(set3dsLoadingStatus(bool));
    };

    setTrustFrameOpenState = (bool) => {
        this.props.dispatch(setTrustFrameOpenState(bool));
    };

    redirectToErrorFrom3ds = () => {
        this.props.dispatch(redirectToErrorFrom3ds());
    };

    setChallengeError = (error) => {
        this.props.dispatch(setChallengeError(error));
    };

    returnToAuth = () => {
        const {authUrl} = this.props;

        redirect(authUrl);
    };

    switchToRestorePassword = () => {
        this.props.dispatch(switchToRestorePassword());
    };

    sendMetrics = (event) => {
        const {challengeType} = this.props;
        const metricsMap = {
            phone: CHALLENGES_PAGE_PHONE_RESTORE,
            phone_confirmation: CHALLENGES_PAGE_PHONE_CALL_RESTORE,
            email: CHALLENGES_PAGE_EMAIL_RESTORE,
            email_code: CHALLENGES_PAGE_EMAIL_CODE_RESTORE
        };

        if (event.target.tagName === 'A') {
            metrics.send([CHALLENGES_PAGE, metricsMap[challengeType]]);
        }
    };

    noteRestoreLinkHandler = (event) => {
        event.preventDefault();

        this.sendMetrics(event);

        if (event.target.tagName === 'A') {
            this.switchToRestorePassword();
        }
    };

    getTitleText = () => {
        const {is3dsChallenge, challengeType, isOneStep3dsExp, hint = ''} = this.props;

        if (is3dsChallenge) {
            return isOneStep3dsExp ? i18n('_AUTH_.challenge.title_3ds_one_step') : i18n('_AUTH_.challenge.title_3ds');
        }

        if (challengeType === 'email_code') {
            const hintString = Array.isArray(hint) ? hint.slice(0, 3).join(', ') : hint;

            return i18n('_AUTH_.challenges.email_code.description').replace('%', hintString);
        }

        if (challengeType === 'question') {
            return i18n('_AUTH_.RestorePwd.question_label');
        }

        return i18n('_AUTH_.challenge.title_sms2fa');
    };

    maybeRenderNote = () => {
        const {isWhiteLabel, isSms2faChallenge, is3dsChallenge, isAm, appId, challengeType} = this.props;

        const modifiedSupportFormLink =
            isAm && appId
                ? `${supportFormLink}${getFirstSymbolForQuery(supportFormLink)}app_id=${appId}`
                : supportFormLink;

        if (isWhiteLabel || is3dsChallenge) {
            return null;
        }

        if (challengeType === 'push_2fa') {
            return (
                <div className={defaultCssClass('note')}>
                    <a onClick={this.sendPush} className='link_theme_normal'>
                        {i18n('_AUTH_.challenges.push-qr-advice-note.send')}
                    </a>{' '}
                    {i18n('_AUTH_.challenges.push-qr-advice-note.other')}
                </div>
            );
        }

        return isSms2faChallenge ? (
            <div
                className={defaultCssClass('note')}
                onClick={this.sendMetrics}
                dangerouslySetInnerHTML={{
                    __html: formatLocTextWithLink(i18n('_AUTH_.challenges.sms2fa-advice-note'), modifiedSupportFormLink)
                }}
            />
        ) : (
            <div
                className={defaultCssClass('note')}
                onClick={this.noteRestoreLinkHandler}
                dangerouslySetInnerHTML={{
                    __html: i18n('_AUTH_.challenges.advice-note').replace('/restoration', '/auth/restore/password')
                }}
            />
        );
    };

    renderError = () => {
        const {errorCode} = this.props;

        return (
            <Page cls='auth-challenge' dataT='auth-challenge'>
                {errorCode === 'challenge.limit_exceeded' ? (
                    <>
                        <Title>{i18n('_AUTH_.challenge.limit_exceeded-title')}</Title>
                        <div className={defaultCssClass('wrapper')}>
                            <div
                                className='auth-challenge-descr'
                                dangerouslySetInnerHTML={{
                                    __html: i18n('_AUTH_.challenge.limit_exceeded').replace(
                                        '%attr',
                                        'class="link_theme_normal"'
                                    )
                                }}
                            />
                        </div>
                    </>
                ) : (
                    <>
                        <Title>{i18n('Errors.session_invalid')}</Title>
                        <div className={defaultCssClass('wrapper')}>
                            <div className='auth-challenge-descr'>
                                <div className='passp-form-field__error'>
                                    {i18n('DeleteData.session.invalid.no-phone')}
                                </div>
                                <div data-t='challenge_reload' className='passp-button'>
                                    <Button
                                        onClick={this.returnToAuth}
                                        text={i18n('DeleteData.try-again')}
                                        view='action'
                                        size='l'
                                        width='max'
                                    />
                                </div>
                            </div>
                        </div>
                    </>
                )}
            </Page>
        );
    };

    renderTitle = () => {
        const {errorCode, is3dsChallenge} = this.props;

        if (errorCode && is3dsChallenge) {
            return null;
        }

        return <Title>{this.getTitleText()}</Title>;
    };

    render() {
        const {
            challengeType,
            hint,
            fieldValue,
            errorText,
            errorCode,
            alternative,
            isSms2faChallenge,
            url3ds,
            isOneStep3dsExp,
            is3dsLoading,
            isTrustFrameOpened
        } = this.props;

        if (errorCode === 'challenge.limit_exceeded' || errorCode === 'internal') {
            return this.renderError();
        }

        return (
            <Page cls='auth-challenge' dataT='auth-challenge'>
                {this.renderTitle()}
                <div className={defaultCssClass('wrapper')}>
                    {challengeType === 'email' && (
                        <ChallengeWithEmail
                            value={fieldValue}
                            passChallenge={this.passChallenge}
                            updateFieldValue={this.updateFieldValue}
                            errorText={errorText}
                            challengeType={challengeType}
                            hint={hint}
                        />
                    )}
                    {challengeType === 'phone' && (
                        <ChallengeWithPhone
                            value={fieldValue}
                            passChallenge={this.passChallenge}
                            updateFieldValue={this.updateFieldValue}
                            errorText={errorText}
                            challengeType={challengeType}
                            hint={hint}
                        />
                    )}
                    {challengeType === 'push_2fa' && (
                        <ChallengeWithPush
                            value={fieldValue}
                            passChallenge={this.passChallenge}
                            sendPush={this.sendPush}
                            updateFieldValue={this.updateFieldValue}
                            errorText={errorText}
                            challengeType={challengeType}
                            hint={hint}
                        />
                    )}
                    {challengeType === 'phone_confirmation' && (
                        <ChallengeWithPhoneConfirmation
                            validatePhone={this.validatePhone}
                            updateFieldValue={this.updateFieldValue}
                            errorText={errorText}
                            isSms2faChallenge={isSms2faChallenge}
                            hint={hint}
                        />
                    )}
                    {challengeType === 'question' && (
                        <ChallengeWithQuestion
                            value={fieldValue}
                            passChallenge={this.passChallenge}
                            updateFieldValue={this.updateFieldValue}
                            errorText={errorText}
                            challengeType={challengeType}
                            hint={hint}
                        />
                    )}
                    {challengeType === '3ds' && (
                        <ChallengeWith3ds
                            submit3ds={this.submit3ds}
                            commit3ds={this.commit3ds}
                            url3ds={url3ds}
                            isOneStep3dsExp={isOneStep3dsExp}
                            redirectToErrorFrom3ds={this.redirectToErrorFrom3ds}
                            is3dsLoading={is3dsLoading}
                            set3dsLoadingStatus={this.set3dsLoadingStatus}
                            setTrustFrameOpenState={this.setTrustFrameOpenState}
                            errorText={errorText}
                            isTrustFrameOpened={isTrustFrameOpened}
                            setChallengeError={this.setChallengeError}
                        />
                    )}
                    {challengeType === 'dictation' && (
                        <ChallengeWithDictation
                            validatePhone={this.validatePhone}
                            updateFieldValue={this.updateFieldValue}
                            errorText={errorText}
                            hint={hint}
                        />
                    )}
                    {challengeType === 'email_code' && (
                        <ChallengeWithEmailCode
                            value={fieldValue}
                            confirmEmailCode={this.confirmEmailCode}
                            sendCodeToEmail={this.sendCodeToEmail}
                            updateFieldValue={this.updateFieldValue}
                            errorText={errorText}
                            hint={hint}
                        />
                    )}
                    {this.maybeRenderNote()}
                </div>
                {alternative && (
                    <DomikFooterLink
                        pseudo={true}
                        href='#'
                        onClick={this.toggleChallenge}
                        dataT='passp:toggle-challenge'
                    >
                        {i18n('_AUTH_.challenge.toggle-another')}
                    </DomikFooterLink>
                )}
            </Page>
        );
    }
}

export default connect(mapStateToProps)(Challenge);

Challenge.propTypes = {
    dispatch: PropTypes.func.isRequired,
    challengeType: PropTypes.string,
    hint: PropTypes.string,
    fieldValue: PropTypes.string,
    errorText: PropTypes.string,
    errorCode: PropTypes.string,
    authUrl: PropTypes.string,
    alternative: PropTypes.object,
    isSms2faChallenge: PropTypes.bool,
    isWhiteLabel: PropTypes.bool,
    is3dsChallenge: PropTypes.bool,
    url3ds: PropTypes.string,
    isOneStep3dsExp: PropTypes.bool,
    is3dsLoading: PropTypes.bool,
    isAm: PropTypes.bool,
    appId: PropTypes.string,
    isTrustFrameOpened: PropTypes.bool
};
