import React, {useState, useRef, useCallback} from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {TYPE_2FA_AUTH} from '@blocks/authv2/constants';
import {Popup} from '@components/Popup';
import {RadioButton} from '@components/RadioButton';
import MagicField from '@blocks/authv2/components/MagicField/MagicField.jsx';
import {LoginLogoId} from '@components/LoginLogoId';
import CaptchaField from '@blocks/authv2/components/CaptchaField/CaptchaField.jsx';
import OTPField from '@blocks/authv2/components/OTPField/OTPField.jsx';
import {SignInButton} from '@blocks/authv2/components/SignInButton';
import {Button} from '@components/Button';
import {ImageIcon} from './ImageIcon';
import {QRIcon} from './QRIcon';
import {OtpIcon} from './OtpIcon';
import {CircleProgress} from './CircleProgress';
import {RefreshIcon} from './RefreshIcon';
import {InfoIcon} from './InfoIcon';

import './Image2FA.styl';

const b = cn('Image2FA');

const ImageOptionContent = () => (
    <div className={b('optionContent')}>
        <div className={b('optionIcon')}>
            <ImageIcon />
        </div>
        <div className={b('optionLabel')}>{i18n('_AUTH_.key2fa.image')}</div>
    </div>
);

const QROptionContent = () => (
    <div className={b('optionContent')}>
        <div className={b('optionIcon')}>
            <QRIcon />
        </div>
        <div className={b('optionLabel')}>{i18n('_AUTH_.key2fa.qr-code')}</div>
    </div>
);

const OtpOptionContent = () => (
    <div className={b('optionContent')}>
        <div className={b('optionIcon')}>
            <OtpIcon />
        </div>
        <div className={b('optionLabel')}>{i18n('_AUTH_.key2fa.password')}</div>
    </div>
);

const ONE_MINUTE = 1000 * 60;
const DELAY_TIME = ONE_MINUTE;

export function Image2FA({
    authUrl,
    attemptsAmount,
    correctImageNumber,
    csrf,
    expiresAt,
    hasCaptcha,
    keyPicturesUrl,
    loading,
    login,
    magicError,
    password,
    trackId,
    type2FA,
    fieldError,
    isMagicSuccess,
    changePasswordValue,
    setType2FA,
    updateImage,
    submitOTP
}) {
    const [tooltipIsVisible, setTooltipIsVisible] = useState(false);
    const [isTimeOver, setTimeOver] = useState(false);

    const tooltipRef = useRef();

    const startTime = expiresAt - DELAY_TIME;

    const submitPasswordForm = useCallback(
        (event) => {
            event.preventDefault();
            submitOTP(password);
        },
        [submitOTP, password]
    );

    return (
        <div className={b('')}>
            <LoginLogoId />
            <div className={b('body')}>
                {type2FA === TYPE_2FA_AUTH.IMAGE && !isTimeOver && !magicError && attemptsAmount > 0 && (
                    <div className={b('content')}>
                        <div className={b('title')}>
                            {i18n('_AUTH_.key2fa.choose-image-in-app')}
                            <span className={b('tooltip')} ref={tooltipRef}>
                                <InfoIcon
                                    onMouseEnter={() => setTooltipIsVisible(true)}
                                    onClick={() => setTooltipIsVisible(true)}
                                    onMouseLeave={() => setTooltipIsVisible(false)}
                                />
                                <Popup
                                    visible={tooltipIsVisible}
                                    directions={['top-center', 'top-left', 'top-right']}
                                    hasTail={true}
                                    target='anchor'
                                    anchor={tooltipRef}
                                    zIndex={9999}
                                    view='black'
                                    onClose={() => setTooltipIsVisible(false)}
                                >
                                    <div className={b('tooltipText')}>{b('_AUTH_.key2fa.version-tip')}</div>
                                </Popup>
                            </span>
                        </div>
                        <div className={b('imageProgress')}>
                            <CircleProgress
                                startTime={startTime}
                                onTimeOver={() => {
                                    setTimeOver(true);
                                }}
                            />
                            <img
                                className={b('image')}
                                src={`${keyPicturesUrl}/${correctImageNumber}/${correctImageNumber}_mdpi.png`}
                            />
                        </div>
                    </div>
                )}
                {type2FA === TYPE_2FA_AUTH.IMAGE && (isTimeOver || magicError) && attemptsAmount > 0 && (
                    <div className={b('content')}>
                        <div className={b('title')}>{i18n('_AUTH_.key2fa.time-is-up')}</div>
                        <div className={b('attemptFail')}>
                            <Button
                                className={b('updateImageBtn')}
                                dataT={'updateImageBtn'}
                                view='rounded-shadow'
                                onClick={() => updateImage(login)}
                                width='auto'
                                size='xxl'
                            >
                                <div className={b('updateImageBtnText')}>
                                    <RefreshIcon />
                                    {i18n('_AUTH_.key2fa.update-image')}
                                </div>
                            </Button>
                            <div className={b('attemptsAmount', {lastAttempt: attemptsAmount === 1})}>
                                {i18n(
                                    {
                                        count: attemptsAmount,
                                        one: '_AUTH_.key2fa.attempts-left.one',
                                        some: '_AUTH_.key2fa.attempts-left.some',
                                        many: '_AUTH_.key2fa.attempts-left.many',
                                        none: '_AUTH_.key2fa.attempts-left.many'
                                    },
                                    attemptsAmount
                                )}
                            </div>
                        </div>
                    </div>
                )}
                {type2FA === TYPE_2FA_AUTH.IMAGE && (isTimeOver || magicError) && attemptsAmount === 0 && (
                    <div className={b('content', {fail: true})}>
                        <img src='/st/i/authv2/image2fa/fail-enter-image-2fa.png' />
                        <div className={b('failText')}>
                            {isTimeOver && attemptsAmount === 0
                                ? i18n('_AUTH_.key2fa.attempts-ended-try-qr-or-otp')
                                : i18n('_AUTH_.key2fa.failed-entry-with-image')}
                        </div>
                    </div>
                )}
                {type2FA === TYPE_2FA_AUTH.QR && (
                    <div className={b('content')}>
                        <div className={b('title')}>{i18n('_AUTH_.key2fa.scan-qr-to-entry')}</div>
                        <MagicField
                            trackId={trackId}
                            fieldError={fieldError}
                            icon={'yandexKey'}
                            className={b('magicField', {hide: isMagicSuccess})}
                        />
                    </div>
                )}
                {type2FA === TYPE_2FA_AUTH.OTP && (
                    <div className={b('content')}>
                        <div className={b('title')}>{i18n('_AUTH_.key2fa.otp')}</div>
                        <form onSubmit={submitPasswordForm} method='post' action={authUrl} noValidate={true}>
                            <input type='hidden' name='track_id' value={trackId} readOnly={true} />
                            <input type='hidden' name='csrf_token' value={csrf} readOnly={true} />
                            {hasCaptcha && (
                                <div className={b('captcha')}>
                                    <CaptchaField />
                                </div>
                            )}
                            <OTPField onChange={changePasswordValue} password={password} />
                            <SignInButton isDisabled={loading} />
                        </form>
                    </div>
                )}
            </div>
            <div className={b('tabs')}>
                <div className={b('enterType')}>{i18n('_AUTH_.key2fa.method-of-entry')}</div>
                <RadioButton
                    width='max'
                    value={type2FA}
                    view='bigRounded'
                    options={[
                        {value: TYPE_2FA_AUTH.IMAGE, children: <ImageOptionContent />},
                        {value: TYPE_2FA_AUTH.QR, children: <QROptionContent />},
                        {value: TYPE_2FA_AUTH.OTP, children: <OtpOptionContent />}
                    ]}
                    onChange={(event) => {
                        if (type2FA !== event.target.value) {
                            setType2FA(event.target.value, login);
                        }
                    }}
                />
            </div>
        </div>
    );
}

Image2FA.propTypes = {
    authUrl: PropTypes.string,
    attemptsAmount: PropTypes.number,
    correctImageNumber: PropTypes.number,
    csrf: PropTypes.string,
    expiresAt: PropTypes.number,
    hasCaptcha: PropTypes.bool,
    keyPicturesUrl: PropTypes.string,
    loading: PropTypes.bool,
    login: PropTypes.string,
    magicError: PropTypes.string,
    password: PropTypes.string,
    trackId: PropTypes.string,
    type2FA: PropTypes.string,
    fieldError: PropTypes.string,
    isMagicSuccess: PropTypes.bool,
    changePasswordValue: PropTypes.func,
    setType2FA: PropTypes.func,
    updateImage: PropTypes.func,
    submitOTP: PropTypes.func
};
