import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Link} from '@components/Link';
import magic from '@blocks/authv2/magic';
import {SHOW_QR_FORM, SHOW_QR_FORM_GOAL} from '@blocks/authv2/metrics_constants';
import metrics from '@blocks/metrics';
import mapStateToProps from './mapStateToProps';
import mapDispatchToProps from './mapDispatchToProps';
import MagicField from '@blocks/authv2/components/MagicField/MagicField.jsx';
import Page from '@blocks/authv2/components/Page/Page.jsx';
import sendApiRequest from '@blocks/authv2/actions/sendApiRequest';
import multiStepAuthStart from '@blocks/authv2/actions/multiStepAuthStart';
import {Divider} from '@blocks/authv2/components/Divider';
import {MagicAvatar} from '@blocks/authv2/components/MagicAvatar';
import {YandexAppButtons} from '@components/YandexAppButtons';
import {Captcha} from '@screens/Captcha';
import {Image2FA} from './Image2FA';

const b = cn('MagicPromoPage');

class MagicPromoPage extends Component {
    constructor(props) {
        super(props);

        this.switchToOTP = this.switchToOTP.bind(this);
    }

    componentDidMount() {
        const {
            magicUrl,
            login,
            setupMode,
            setupPane,
            authPasswordSubmitForMagic,
            multiStepAuthStartForMagic
        } = this.props;

        setupMode('magic');
        setupPane(magicUrl);

        metrics.goal(SHOW_QR_FORM_GOAL);
        metrics.send([SHOW_QR_FORM]);

        if (login) {
            multiStepAuthStartForMagic(login);
        }

        authPasswordSubmitForMagic();
    }

    componentDidUpdate(prevProps) {
        if (prevProps.magicCSRFToken !== this.props.magicCSRFToken) {
            magic.restart();
            return;
        }

        if (prevProps.trackId !== this.props.trackId) {
            magic.restart();
        }
    }

    componentWillUnmount() {
        magic.clearCaptchaState();
        magic.stop();
    }

    switchToOTP(event) {
        event.preventDefault();

        const {
            switchToModeAddingAccount,
            forceOTPMethod,
            login,
            addUserUrl,
            editUrl,
            isSocialAccount,
            dispatch
        } = this.props;

        if (!login) {
            switchToModeAddingAccount();
            return;
        }

        forceOTPMethod(true);

        const backPane = isSocialAccount ? editUrl : addUserUrl;

        dispatch(sendApiRequest(multiStepAuthStart, {login, backPane}));
    }

    onCaptchaCheck = () => {
        magic.clearCaptchaState();
        magic.restart();
    };

    maybeRenderCaptcha = () => {
        const {isCaptchaRequired} = this.props;

        if (!isCaptchaRequired) {
            return null;
        }

        return <Captcha onConfirmCallback={this.onCaptchaCheck} />;
    };

    maybeRenderQR = () => {
        const {
            trackId,
            fieldError,
            addUserUrl,
            fallbackLinkText,
            texts,
            is2faWithImage,
            isWebView,
            isAccountWith2FA,
            language,
            isCaptchaRequired,
            isMagicSuccess
        } = this.props;
        const {descriptionText} = texts;

        if (isCaptchaRequired || is2faWithImage) {
            return null;
        }

        const iconLang = !language || ['ru', 'uk'].includes(language) ? 'ru' : 'en';

        return (
            <>
                <h1 className={b('title')} dangerouslySetInnerHTML={{__html: descriptionText}} />

                <div className={b('magic')}>
                    {isMagicSuccess && <MagicAvatar />}
                    <MagicField
                        trackId={trackId}
                        fieldError={fieldError}
                        icon={isAccountWith2FA ? 'yandexKey' : `yandex3-${iconLang}`}
                        className={b('magicField', {hide: isMagicSuccess})}
                    />
                    <div className={b('controls', {hide: isMagicSuccess})}>
                        <Link href={addUserUrl} onClick={this.switchToOTP} weight='medium'>
                            {fallbackLinkText}
                        </Link>
                    </div>
                </div>

                {!isAccountWith2FA && (
                    <>
                        <Divider />
                        {!isWebView && <YandexAppButtons />}
                    </>
                )}
            </>
        );
    };

    maybeRenderImage2FA = () => {
        const {is2faWithImage, isCaptchaRequired} = this.props;

        if (isCaptchaRequired || !is2faWithImage) {
            return null;
        }

        return <Image2FA />;
    };

    render() {
        return (
            <Page cls='magic' dataT='magic'>
                {this.maybeRenderCaptcha()}
                {this.maybeRenderQR()}
                {this.maybeRenderImage2FA()}
            </Page>
        );
    }
}

MagicPromoPage.propTypes = {
    addUserUrl: PropTypes.string.isRequired,
    editUrl: PropTypes.string.isRequired,
    magicUrl: PropTypes.string.isRequired,
    trackId: PropTypes.string,
    magicCSRFToken: PropTypes.string,
    tld: PropTypes.string.isRequired,
    fallbackLinkText: PropTypes.string.isRequired,
    multiStepAuthStartForMagic: PropTypes.func.isRequired,
    login: PropTypes.string,
    language: PropTypes.string,
    is2faWithImage: PropTypes.bool,
    isTouch: PropTypes.bool.isRequired,
    isSocialAccount: PropTypes.bool.isRequired,
    fieldError: PropTypes.string,
    setupMode: PropTypes.func.isRequired,
    setupPane: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    authPasswordSubmitForMagic: PropTypes.func.isRequired,
    switchToModeAddingAccount: PropTypes.func.isRequired,
    forceOTPMethod: PropTypes.func.isRequired,
    texts: PropTypes.object,
    isAccountWith2FA: PropTypes.bool,
    isWebView: PropTypes.bool,
    isCaptchaRequired: PropTypes.bool,
    isMagicSuccess: PropTypes.bool
};

export default connect(mapStateToProps, mapDispatchToProps)(MagicPromoPage);
