import React from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {isKeyEnabled2FASelector} from '@blocks/selectors';
import EnterPhoneForm from '@blocks/authv2/components/EnterPhoneForm/EnterPhoneForm.jsx';
import Page from '@blocks/authv2/components/Page/Page.jsx';
import Title from '@blocks/authv2/components/Title/Title.jsx';
import {Button} from '@components/Button';
import Form from '@blocks/authv2/components/Form/Form.jsx';
import {
    updatePhoneNumber,
    sendConfirmationCode,
    clearSecurePhoneNumber,
    setConfirmProcessState
} from '@blocks/actions/phoneConfirm';
import errorMessages from '@blocks/authv2/errors';
import {removePhone} from '@blocks/authv2/actions/nativeMobileApi';

class PhoneConfirm extends React.PureComponent {
    _handlePhoneChange = (value) => {
        const {updatePhoneNumber} = this.props;

        updatePhoneNumber(value);
    };

    _sendConfirmationCode = (event) => {
        const {sendConfirmationCode} = this.props;

        event.preventDefault();
        sendConfirmationCode();
    };

    _sendConfirmationCodeForSecurePhone = (event) => {
        const {sendConfirmationCode, setConfirmProcessState} = this.props;

        event.preventDefault();
        setConfirmProcessState(true);
        sendConfirmationCode();
    };

    _clearSecurePhoneNumber = (event) => {
        const {clearSecurePhoneNumber} = this.props;

        event.preventDefault();
        clearSecurePhoneNumber();
    };

    _getClearFormTitle = () => {
        const {isPhoneConfirmationForChildishReg, isKeyEnabled2FA} = this.props;

        if (isPhoneConfirmationForChildishReg) {
            return i18n('_AUTH_.phone-confirm.phone.title.childish');
        }

        if (isKeyEnabled2FA) {
            return i18n('_AUTH_.phone-confirm.phone.title.key-2fa');
        }

        return i18n('_AUTH_.phone-confirm.phone.title');
    };

    _maybeRenderClearForm = () => {
        const {
            isCaptchaRequired,
            errors,
            phoneNumber,
            securePhoneNumber,
            isSecurePhoneMode,
            isAm,
            amPhoneNumber
        } = this.props;
        const isAmWithPhone = isAm && amPhoneNumber;
        const isAmPhoneSecure = isAmWithPhone && amPhoneNumber === securePhoneNumber;

        if (securePhoneNumber || isAmWithPhone) {
            return null;
        }

        return (
            <React.Fragment>
                <Title>{this._getClearFormTitle()}</Title>

                <EnterPhoneForm
                    handlePhoneChange={this._handlePhoneChange}
                    sentConfirmationCode={
                        isAmPhoneSecure ? this._sendConfirmationCodeForSecurePhone : this._sendConfirmationCode
                    }
                    phoneNumber={phoneNumber}
                    isCaptchaRequired={isCaptchaRequired}
                    clearPhoneForm={this._clearPhoneForm}
                    action='add'
                    errors={errors}
                    isSecurePhoneMode={isSecurePhoneMode}
                />
            </React.Fragment>
        );
    };

    _maybeRenderSecurePhoneConfirm = () => {
        const {
            securePhoneNumber,
            errors,
            isSecurePhoneMode,
            profilePhonesUrl,
            isAm,
            isKeyEnabled2FA,
            amEditable,
            amPhoneNumber
        } = this.props;
        const isAmWithPhone = isAm && amPhoneNumber;
        const isAmPhoneSecure = isAm && amPhoneNumber === securePhoneNumber;

        let error;

        if (errors.length > 0) {
            error = errorMessages[errors[0]] ? errorMessages[errors[0]] : errorMessages.internal;
        }

        if (!securePhoneNumber && !isAmWithPhone) {
            return null;
        }

        const phoneToShow = isAmWithPhone ? amPhoneNumber : securePhoneNumber;
        const phoneToShowTexts = i18n('_AUTH_.phone-confirm.phone.title-with-number').split('%phone');

        const submitButtonText = isKeyEnabled2FA
            ? i18n('Frontend.send_sms')
            : i18n('_AUTH_.phone.enter-confirmation-code.submit');

        return (
            <React.Fragment>
                {isKeyEnabled2FA ? (
                    <Title>
                        {i18n('_AUTH_.phone-confirm.phone.title-with-number.key-2fa').replace('%phone', phoneToShow)}
                    </Title>
                ) : (
                    <Title>
                        <span
                            dangerouslySetInnerHTML={{
                                __html: phoneToShowTexts[0]
                            }}
                        />
                        <br />
                        <b>{phoneToShow}</b>
                        <span
                            dangerouslySetInnerHTML={{
                                __html: phoneToShowTexts[1]
                            }}
                        />
                    </Title>
                )}

                <Form dataT='phone-confirm'>
                    {error && <div className='passp-form-field__error' dangerouslySetInnerHTML={{__html: error}} />}

                    <div data-t='phone_submit' className='passp-button'>
                        <Button
                            type='submit'
                            text={submitButtonText}
                            view='action'
                            width='max'
                            size='l'
                            onClick={
                                isAmWithPhone && !isAmPhoneSecure
                                    ? this._sendConfirmationCode
                                    : this._sendConfirmationCodeForSecurePhone
                            }
                        />
                    </div>

                    {(!isAm || amEditable) && (
                        <div data-t='phone_reset' className='passp-button'>
                            <Button
                                type={isSecurePhoneMode ? 'link' : 'submit'}
                                text={i18n('_AUTH_.phone-confirm.phone.another-number')}
                                view='pseudo'
                                width='max'
                                size='l'
                                url={isSecurePhoneMode ? profilePhonesUrl : undefined}
                                onClick={isSecurePhoneMode ? undefined : this._clearSecurePhoneNumber}
                            />
                        </div>
                    )}
                </Form>
            </React.Fragment>
        );
    };

    render() {
        const {trackId} = this.props;

        return (
            <Page cls='phone-request' dataT='phone-request'>
                <input data-t='track_id' type='hidden' value={trackId} />

                {this._maybeRenderClearForm()}
                {this._maybeRenderSecurePhoneConfirm()}
            </Page>
        );
    }
}

PhoneConfirm.propTypes = {
    isCaptchaRequired: PropTypes.bool,
    isAm: PropTypes.bool,
    amPhoneNumber: PropTypes.string,
    amEditable: PropTypes.bool,
    phoneNumber: PropTypes.string,
    securePhoneNumber: PropTypes.string,
    trackId: PropTypes.string.isRequired,
    profilePhonesUrl: PropTypes.string.isRequired,
    errors: PropTypes.array,
    updatePhoneNumber: PropTypes.func.isRequired,
    sendConfirmationCode: PropTypes.func.isRequired,
    clearSecurePhoneNumber: PropTypes.func.isRequired,
    setConfirmProcessState: PropTypes.func.isRequired,
    isSecurePhoneMode: PropTypes.bool,
    isPhoneConfirmationForChildishReg: PropTypes.bool,
    isKeyEnabled2FA: PropTypes.bool
};

const mapStateToProps = (state = {}) => {
    const {phoneConfirm = {}, common = {}, am = {}} = state;
    const {track_id: trackId, profilePhonesUrl, origin} = common;

    return {
        ...phoneConfirm,
        profilePhonesUrl,
        trackId,
        isAm: am.isAm,
        amEditable: am.editable,
        amPhoneNumber: am.phone,
        isPhoneConfirmationForChildishReg: origin === 'need_access_to_childish_registration',
        isKeyEnabled2FA: isKeyEnabled2FASelector(state)
    };
};

function clearSecurePhoneNumberThunk() {
    return (dispatch) => {
        dispatch(updatePhoneNumber(''));
        dispatch(clearSecurePhoneNumber());
        dispatch(removePhone());
    };
}

const mapDispatchToProps = {
    updatePhoneNumber,
    sendConfirmationCode,
    clearSecurePhoneNumber: clearSecurePhoneNumberThunk,
    setConfirmProcessState
};

export default connect(mapStateToProps, mapDispatchToProps)(PhoneConfirm);
