import defineConvertOptions from '../utils/defineConvertOptions';
import {avatarCropper} from '../utils/avatarCroper';
import {MAX_FILE_SIZE_BYTES, MIN_IMAGE_RANGE} from '../avatarConsts';
import handleError from '../utils/setErrors';
import {setAvatarProcessingStatus, setLoadingStatus, updateAvatarUrl, updateImageOptions, setAvatarError} from '.';
import sendMetrics from '../utils/sendMetrics';

export default function loadAvatarByFile(file, isCropNotSupported) {
    return (dispatch, getState) => {
        const state = getState();
        const {
            avatar: {avatarSize_300: currentUrl},
            settings
        } = state;

        dispatch(setAvatarError(''));

        if (!file) {
            return;
        }

        dispatch(setLoadingStatus(true));
        dispatch(
            updateAvatarUrl({
                prevUrl: currentUrl || settings.avatar.default_300
            })
        );
        sendMetrics('Редактировать аватар', 'avatar-displayname_change_avatar');

        const allowedFileTypes = ['png', 'jpg', 'jpeg', 'gif'];
        // eslint-disable-next-line compat/compat
        const reader = new FileReader();
        const fileSplitted = file.type.split('/');
        const fileType = fileSplitted[0];
        const imageType = fileSplitted[1];
        const isNotImage = fileType !== 'image';
        const isSizeTooBig = file.size > MAX_FILE_SIZE_BYTES;
        const fileInfo = defineConvertOptions(imageType, file.size);

        if (isNotImage || !allowedFileTypes.includes(imageType)) {
            dispatch(handleError('error-file.invalid'));
            return;
        }

        if (isSizeTooBig) {
            dispatch(handleError('error-file.size'));
            return;
        }

        dispatch(
            updateImageOptions({
                type: fileInfo.type,
                compression: fileInfo.compression
            })
        );

        reader.onload = (event) => {
            const testImg = new Image();

            testImg.src = event.target.result;
            testImg.onload = () => {
                const width = testImg.naturalWidth || testImg.width;
                const height = testImg.naturalHeight || testImg.height;
                const isNotAppropriateSize = width < MIN_IMAGE_RANGE || height < MIN_IMAGE_RANGE;

                if (isNotAppropriateSize) {
                    dispatch(handleError('error-image.size'));
                    return;
                }
                dispatch(
                    updateAvatarUrl({
                        url: event.target.result,
                        prevUrl: currentUrl
                    })
                );

                if (isCropNotSupported) {
                    const form = document.getElementById('load_avatar');

                    if (form) {
                        return form.submit();
                    }

                    return dispatch(handleError('error-internal'));
                }

                avatarCropper.init();
                dispatch(setLoadingStatus(false));
                dispatch(setAvatarProcessingStatus(true));
            };
        };

        reader.onerror = () => {
            dispatch(handleError('error-internal'));
        };

        reader.readAsDataURL(file);
    };
}
