import api from '@blocks/api';
import processErrors from '@blocks/avatar-displayname/utils/processErrors';
import {setDisplayNameError, setDisplayNameLoadingStatus} from './';
import {setDisplayName as setHeaderDisplayName} from '@blocks/morda/components/header/actions';
import metrics from '@blocks/metrics';
import {getTrackId, getUid, getProcessUUid, getRedirectUrl, getFormValues, getFormErrors} from '@blocks/selectors';
import {getError, sendMessage} from '@blocks/utils';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import {updateErrors} from '@blocks/actions/form';

export default function updateDisplayName() {
    return (dispatch, getState) => {
        const state = getState();
        const {
            displayName: {value, publicDisplayName, isFromVariants, displayNamesList, isProcessing},
            publicId = {}
        } = state;
        const retpath = getRedirectUrl(state);
        const uid = getUid(state);
        const trackId = getTrackId(state, TRACK_TYPES.COMMON);
        const form = getFormValues(state);
        const formErrors = getFormErrors(state);
        const params = {
            track_id: trackId,
            display_name: isFromVariants ? displayNamesList[value] : value,
            is_from_variants: isFromVariants,
            process_uuid: getProcessUUid(state),
            stat_uid: uid,
            stat_mod: 'public_avatar_displayname'
        };

        if (!value && isProcessing) {
            dispatch(setDisplayNameError(processErrors('display_name.empty')));
            return;
        }

        if (value === publicDisplayName && !form.publicId) {
            sendMessage({updateDisplayName: 'success'}, retpath);
            location.href = retpath;
            return;
        }

        if (form.publicId) {
            params.public_id = form.publicId;
        }

        if (formErrors && formErrors.publicId && formErrors.publicId.code) {
            return;
        }

        dispatch(setDisplayNameLoadingStatus(true));

        api.request('update.profile', params)
            .done((response = {}) => {
                if (response.status === 'ok') {
                    sendMessage({updateDisplayName: 'success'}, retpath);
                    dispatch(setHeaderDisplayName(uid, value));
                    metrics.send(['Изменить персональные данные', 'Сохранить display name']);
                    api.writeStatbox({
                        mode: 'public_avatar_displayname',
                        track_id: trackId,
                        action: 'update'
                    });
                    location.href = retpath;
                } else {
                    sendMessage({updateDisplayName: 'error'}, retpath);
                    dispatch(setDisplayNameError(processErrors('error-internal')));
                }
            })
            .fail((error = {}) => {
                const {status, errors = []} = error;

                sendMessage({updateDisplayName: 'error'}, retpath);
                if (status === 'error') {
                    errors.forEach((code) => {
                        if (code.startsWith('public_id')) {
                            dispatch(updateErrors({field: 'publicId', error: getError('publicId', code)}));
                        } else {
                            dispatch(setDisplayNameError(processErrors([code])));
                        }
                    });
                } else {
                    dispatch(setDisplayNameError(processErrors('error-internal')));
                }
                metrics.send(['Изменить персональные данные', 'Сохранить display name ошибка']);
            })
            .always(() => {
                if (publicId.updatesNumber === 2) {
                    dispatch(
                        updateErrors({field: 'publicId', error: getError('publicId', 'public_id.update_not_allowed')})
                    );
                }

                dispatch(setDisplayNameLoadingStatus(false));
            });
    };
}
