import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {hasExp, isMobile} from '@blocks/selectors';

import {Spin} from '@components/Spin';
import {Button} from '@components/Button';
import {Attach} from '@components/Attach';

import mapDispatchToProps from './mapDispatchToProps';

import Error from '../Error/Error.jsx';
import SimpleForm from './SimpleForm.jsx';
import loadAvatarByFile from '../../actions/loadAvatarByFile';
import setErrors from '../../utils/setErrors';
import {setAvatarError, setDisplayNameProcessingStatus} from '../../actions';

const b = cn('Avatar');

class Avatar extends Component {
    constructor(props) {
        super(props);

        this.handleFile = this.handleFile.bind(this);
    }

    handleFile(event) {
        const {dispatch} = this.props;
        const file = event.target.files && event.target.files[0];

        dispatch(setDisplayNameProcessingStatus(false));

        if (window.File && window.FileReader) {
            dispatch(loadAvatarByFile(file));
        } else {
            dispatch(setErrors('error-internal'));
        }
    }

    componentDidUpdate(prevProps) {
        const {error: prevError} = prevProps;
        const {error, onPropsUpdate} = this.props;

        if (error !== prevError && typeof onPropsUpdate === 'function') {
            onPropsUpdate();
        }
    }

    renderUpdatedAvatarBlock = () => {
        const {
            default300,
            avatarImage,
            isProcessing,
            isLoading,
            isTouch,
            error,
            cropAvatar,
            cancelCrop,
            isCropNotSupported,
            dispatch
        } = this.props;

        if (isCropNotSupported) {
            return <SimpleForm isTouch={isTouch} />;
        }

        return (
            <div className={b('container')}>
                <div className={b('imageWrapSimple')}>
                    <div
                        className={b('image', {
                            isProcessing,
                            borderRadius: !isProcessing,
                            avatarDefault: default300 === avatarImage
                        })}
                    >
                        <img src={avatarImage} alt='user avatar' id='user-avatar' className={b('imageElem')} />
                        {!isProcessing && (
                            <>
                                <span className={b('addPhotoIcon')} />
                                <input
                                    type='file'
                                    name='passp_load_avatar'
                                    id='passp_load_avatar'
                                    className={b('attach')}
                                    onChange={this.handleFile}
                                />
                            </>
                        )}
                        <div className={b('spin')}>
                            <Spin size='s' progress={isLoading} />
                        </div>
                    </div>
                    <div className={b('controls')}>
                        {isProcessing && (
                            <div className={b('controlBtns')}>
                                <Button view='pseudo' size='s' disabled={isLoading} onClick={cropAvatar}>
                                    {i18n('Frontend.ready')}
                                </Button>
                                <Button view='pseudo' size='s' disabled={isLoading} onClick={cancelCrop}>
                                    {i18n('_AUTH_.cancel.operation')}
                                </Button>
                            </div>
                        )}
                    </div>
                </div>
                <Error
                    isTouch={isTouch}
                    isVisible={Boolean(error)}
                    text={error}
                    dispatch={dispatch}
                    hideErrorAction={setAvatarError}
                />
            </div>
        );
    };

    getDescription = () => {
        const {default300, avatarImage, isProcessing, isMobile, deleteAvatar} = this.props;
        const hasAvatar = default300 !== avatarImage && !isProcessing;

        if (hasAvatar) {
            return (
                <>
                    <p className={b('descriptionText')}>{i18n('Profile.avatar_set.description')}</p>
                    {!isProcessing && (
                        <div className={b('attachWrapper')}>
                            <Attach view='clear' hasHolder={false} onChange={this.handleFile}>
                                {i18n('Profile.avatar_set.button')}
                            </Attach>
                        </div>
                    )}
                    {isMobile && (
                        <div className={b('deleteMobile')}>
                            <Button view='pseudo' type='button' size='s' onClick={deleteAvatar}>
                                {i18n('Profile.common.delete')}
                            </Button>
                        </div>
                    )}
                </>
            );
        }

        return (
            <>
                <p className={b('descriptionText')}>{i18n('Profile.avatar_set.description-no-avatar')}</p>
                {!isProcessing && (
                    <Attach
                        view='default'
                        size='m'
                        disabled={false}
                        hasHolder={false}
                        onClearClick={this.clearFile}
                        onChange={this.handleFile}
                    >
                        {i18n('Profile2.add.photo')}
                    </Attach>
                )}
            </>
        );
    };

    render() {
        const {
            default300,
            avatarImage,
            isProcessing,
            isLoading,
            isTouch,
            error,
            deleteAvatar,
            cropAvatar,
            cancelCrop,
            isCropNotSupported,
            isMobile,
            isPublicAllowCheckbox,
            dispatch
        } = this.props;
        const isDeleteBtnShowed = default300 !== avatarImage && !isProcessing;

        if (isCropNotSupported) {
            return <SimpleForm isTouch={isTouch} />;
        }

        if (isPublicAllowCheckbox) {
            return this.renderUpdatedAvatarBlock();
        }

        return (
            <div className={b('container')}>
                <div className={b('imageWrap')}>
                    <div
                        className={b('image', {borderRadius: !isProcessing, avatarDefault: default300 === avatarImage})}
                    >
                        <img src={avatarImage} alt='user avatar' id='user-avatar' />
                        {!isProcessing && (
                            <>
                                <div className={b('photoIcon')} />
                                <span className={b('photoIconPic')} />
                                <input
                                    type='file'
                                    name='passp_load_avatar'
                                    id='passp_load_avatar'
                                    className={b('attach')}
                                    onChange={this.handleFile}
                                />
                            </>
                        )}
                        <div className={b('spin')}>
                            <Spin size='s' progress={isLoading} />
                        </div>
                        {!isMobile && isDeleteBtnShowed && (
                            <button
                                type='button'
                                className={b('delete')}
                                onClick={deleteAvatar}
                                title={i18n('Profile.common.delete')}
                            />
                        )}
                    </div>
                    <div className={b('controls')}>
                        <div className={b('description')}>{this.getDescription()}</div>
                        {isProcessing && (
                            <div className={b('controlBtns')}>
                                <Button view='pseudo' size='s' disabled={isLoading} onClick={cropAvatar}>
                                    {i18n('Frontend.ready')}
                                </Button>
                                <Button view='pseudo' size='s' disabled={isLoading} onClick={cancelCrop}>
                                    {i18n('_AUTH_.cancel.operation')}
                                </Button>
                            </div>
                        )}
                    </div>
                </div>
                <Error
                    isTouch={isTouch}
                    isVisible={Boolean(error)}
                    text={error}
                    dispatch={dispatch}
                    hideErrorAction={setAvatarError}
                />
            </div>
        );
    }
}

export default connect(mapStateToProps, mapDispatchToProps)(Avatar);

function mapStateToProps(state) {
    const {
        avatar: {
            default_300: default300,
            avatarSize_300: avatarImage,
            isProcessing,
            isLoading,
            error,
            deleteAvatar,
            cropAvatar,
            cancelCrop,
            isCropNotSupported
        }
    } = state;

    return {
        default300,
        avatarImage,
        isProcessing,
        isLoading,
        error,
        deleteAvatar,
        cropAvatar,
        cancelCrop,
        isCropNotSupported,
        isMobile: isMobile(state),
        isPublicAllowCheckbox: hasExp(state, 'personal_data_checkbox_on')
    };
}

Avatar.propTypes = {
    default300: PropTypes.string.isRequired,
    avatarImage: PropTypes.string.isRequired,
    isProcessing: PropTypes.bool.isRequired,
    isLoading: PropTypes.bool.isRequired,
    error: PropTypes.string,
    deleteAvatar: PropTypes.func.isRequired,
    cropAvatar: PropTypes.func.isRequired,
    cancelCrop: PropTypes.func.isRequired,
    isCropNotSupported: PropTypes.bool.isRequired,
    dispatch: PropTypes.func.isRequired,
    onPropsUpdate: PropTypes.func,
    isTouch: PropTypes.bool.isRequired,
    isMobile: PropTypes.bool,
    isPublicAllowCheckbox: PropTypes.bool
};
