import React, {Component} from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {focusInput} from '@blocks/utils';
import {cn} from '@bem-react/classname';
import {Input} from '@components/Input';
import {Link} from '@components/Link';
import {Popup} from '@components/Popup';
import {Spin} from '@components/Spin';
import {hasExp} from '@blocks/selectors';
import mapDispatchToProps from './mapDispatchToProps';

import {
    setDisplayNameError,
    updateDisplayNameValue,
    setDisplayNameProcessingStatus,
    setDisplayNameFromVariants,
    updateSuggestedDisplaynames
} from '../../actions';

import Error from '../Error/Error.jsx';

const b = cn('PasspDisplayName');

class DisplayName extends Component {
    suggestAnchor = React.createRef();
    hintAnchor = React.createRef();

    state = {
        isSuggestVisible: false,
        isHintVisible: false
    };

    componentDidMount() {
        const {getDisplayName} = this.props;

        getDisplayName();

        if (this.inputRef) {
            const input = this.inputRef.querySelector('input');

            if (input) {
                focusInput(input);
            }
        }
    }

    componentDidUpdate(prevProps, prevState) {
        const {
            displayName: {error: prevError, filteredSuggestedNames: prevFilteredSuggestedNames = []}
        } = prevProps;
        const {
            displayName: {error, filteredSuggestedNames = []},
            onPropsUpdate
        } = this.props;
        const {isSuggestVisible: prevIsSuggestVisible} = prevState;
        const {isSuggestVisible} = this.state;

        if (
            (error !== prevError ||
                filteredSuggestedNames.length !== prevFilteredSuggestedNames.length ||
                isSuggestVisible !== prevIsSuggestVisible) &&
            typeof onPropsUpdate === 'function'
        ) {
            onPropsUpdate();
        }
    }

    handleValue = (event) => {
        const {target = {}} = event;
        const {value} = target;
        const {
            dispatch,
            displayName: {suggestedNames}
        } = this.props;

        dispatch(updateDisplayNameValue(value));
        dispatch(setDisplayNameError(''));

        if (!value) {
            dispatch(updateSuggestedDisplaynames(suggestedNames));
            this.showSuggest(true);
            return;
        }

        this.checkUserInput(value);
    };

    checkUserInput = (data) => {
        const {
            dispatch,
            displayName: {suggestedNames}
        } = this.props;
        const filteredNamesList = suggestedNames.filter((name) => name.toLowerCase().startsWith(data.toLowerCase()));

        dispatch(updateSuggestedDisplaynames(filteredNamesList));

        if (filteredNamesList.length > 0) {
            this.showSuggest(true);
        }
    };

    setEditingStarted = () => {
        const {
            dispatch,
            displayName: {value}
        } = this.props;

        dispatch(setDisplayNameProcessingStatus(true));
        dispatch(setDisplayNameFromVariants(false));
        this.checkUserInput(value);
    };

    selectNewDisplayName = (event) => {
        const {dispatch} = this.props;
        const name = event.currentTarget && event.currentTarget.dataset && event.currentTarget.dataset.name;
        const isEnterPressed = event.keyCode === 13;

        if (isEnterPressed || event.type === 'click') {
            dispatch(updateDisplayNameValue(name));
            dispatch(setDisplayNameFromVariants(true));
            this.showSuggest(false);
        }
    };

    showSuggest = (isVisible) => {
        this.setState({
            isSuggestVisible: isVisible
        });
    };

    closeSuggest = (event) => {
        const target = event && event.target;

        if (target && target.getAttribute('id') === 'display_name') {
            return;
        }

        this.showSuggest(false);
    };

    toggleHint = () => {
        this.setState({
            isHintVisible: !this.state.isHintVisible
        });
    };

    setSuggestedPublicName = () => {
        const {
            displayName: {suggestedPublicName},
            dispatch
        } = this.props;

        dispatch(updateDisplayNameValue(suggestedPublicName));
    };

    render() {
        const {
            displayName: {suggestedPublicName, value, filteredSuggestedNames, error, isLoading},
            dispatch,
            isTouch,
            isPublicAllowCheckbox
        } = this.props;

        return (
            <div className={b('container')}>
                {isPublicAllowCheckbox && (
                    <div className={b('popupDescr')} ref={this.hintAnchor}>
                        {i18n('Profile2.publicName.hint')}{' '}
                        <span onMouseEnter={this.toggleHint} className={b('popupDescrIcon')} />
                        <Popup
                            view='black'
                            scope={this.hintAnchor}
                            position={{bottom: 30, left: 70}}
                            hasTail={true}
                            visible={true}
                        >
                            <div className={b('popupDescrContent')}>{i18n('Profile2.publicName.hint-text')}</div>
                        </Popup>
                    </div>
                )}
                <div className={b('input')} ref={this.suggestAnchor}>
                    <Input
                        size='l'
                        id='display_name'
                        name='display_name'
                        autoComplete='off'
                        hasClear={true}
                        placeholder={i18n('Profile.displayname.label')}
                        value={value}
                        onFocus={this.setEditingStarted}
                        onChange={this.handleValue}
                        innerRef={(ref) => (this.inputRef = ref)}
                    />
                    <div className={b('inputSpin')}>
                        <Spin size='s' progress={isLoading} />
                    </div>
                    <Error
                        isTouch={isTouch}
                        dispatch={dispatch}
                        hideErrorAction={setDisplayNameError}
                        isVisible={Boolean(error)}
                        text={error}
                    />
                    {suggestedPublicName && (
                        <>
                            <div className={b('optionName')}>
                                {i18n('Profile.displayname.fullname_suggest')}
                                <Link onClick={this.setSuggestedPublicName}>{` ${suggestedPublicName}`}</Link>
                            </div>
                        </>
                    )}
                    {!isPublicAllowCheckbox && <p className={b('descr')}>{i18n('Profile.displayname.description')}</p>}

                    <Popup
                        scope={this.suggestAnchor}
                        position={{top: 44, left: 0}}
                        visible={filteredSuggestedNames.length && this.state.isSuggestVisible && !error}
                        directions={['bottom-left']}
                        onOutsideClick={this.closeSuggest}
                    >
                        <div>
                            <ul className={b('suggestList')}>
                                {filteredSuggestedNames.map((name, index) => {
                                    return (
                                        <li
                                            key={`login-${index}`}
                                            tabIndex='0'
                                            className='link_theme_normal'
                                            data-name={name}
                                            onClick={this.selectNewDisplayName}
                                            onKeyUp={this.selectNewDisplayName}
                                        >
                                            {name}
                                        </li>
                                    );
                                })}
                            </ul>
                        </div>
                    </Popup>
                </div>
            </div>
        );
    }
}

export default connect(mapStateToProps, mapDispatchToProps)(DisplayName);

function mapStateToProps(state) {
    const {displayName = {}, settings = {}, publicId = {}} = state;
    const {tld} = settings;
    const {isEnabled} = publicId;

    return {
        displayName,
        tld,
        isPublicIdEnabled: isEnabled,
        isPublicAllowCheckbox: hasExp(state, 'personal_data_checkbox_on')
    };
}

DisplayName.propTypes = {
    displayName: PropTypes.shape({
        value: PropTypes.string.isRequired,
        suggestedPublicName: PropTypes.string,
        publicDisplayName: PropTypes.string,
        displayNamesList: PropTypes.object,
        suggestedNames: PropTypes.array.isRequired,
        error: PropTypes.string,
        isProcessing: PropTypes.bool.isRequired,
        isLoading: PropTypes.bool.isRequired,
        filteredSuggestedNames: PropTypes.array,
        fullname: PropTypes.string
    }),
    setDisplayName: PropTypes.func.isRequired,
    getDisplayName: PropTypes.func.isRequired,
    tld: PropTypes.string,
    dispatch: PropTypes.func.isRequired,
    onPropsUpdate: PropTypes.func,
    suggestId: PropTypes.string.isRequired,
    isTouch: PropTypes.bool.isRequired,
    isPublicIdEnabled: PropTypes.bool,
    isPublicAllowCheckbox: PropTypes.bool
};
