import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import {mapDispatchToProps} from './mapDispatchToProps';
import {getFormValues, hasExp} from '@blocks/selectors';
import Avatar from '../Avatar/Avatar.jsx';
import DisplayName from '../DisplayName/DisplayName.jsx';
import {PublicId} from '@screens/PublicId';
import {sendMessage} from '../../../utils';
import {Button} from '@components/Button';
import {PersonalDataAllow} from '../PersonalDataAllow/PersonalDataAllow.jsx';
import {PersonalDataAllowPopup} from '../PersonalDataAllowPopup/PersonalDataAllowPopup.jsx';

const WRAPPER_CLASSNAME = 'passp-avatar_displayname__wrapper';
const SUGGEST_ID = 'displaynames_suggest';
const SUGGEST_SHIFT = 10;

class User extends Component {
    defaultWrapperHeight = 0;
    wrapper = null;

    constructor(props) {
        super(props);

        this.state = {};
    }

    componentDidMount() {
        const {retpath} = this.props;

        this.wrapper = document.querySelector(`.${WRAPPER_CLASSNAME}`);
        this.defaultWrapperHeight = this.wrapper.offsetHeight;

        sendMessage({formHeight: this.defaultWrapperHeight}, retpath);
    }

    handleSubmit = (event) => {
        const {
            isPublicAllowCheckbox,
            isPublicAccessPopupVisible,
            hasPublicProfile,
            showPublicAccessPopup,
            handleUpdates,
            togglePersonalDataPermission
        } = this.props;

        if (event) {
            event.preventDefault();
        }

        if (isPublicAllowCheckbox) {
            if (!hasPublicProfile && !isPublicAccessPopupVisible) {
                showPublicAccessPopup();
                return;
            }

            togglePersonalDataPermission();
            return;
        }

        handleUpdates();
    };

    handlePublicAllowPopup = () => {
        const {toggleProfilePublicAccess, togglePersonalDataPermission} = this.props;

        toggleProfilePublicAccess();
        togglePersonalDataPermission();
    };

    closePersonalDataAllowPopup = () => {
        const {showPublicAccessPopup, togglePersonalDataPermission} = this.props;

        showPublicAccessPopup();
        togglePersonalDataPermission();
    };

    getHeightWithSuggest = () => {
        const suggest = document.querySelector(`#${SUGGEST_ID}`);

        if (!suggest) {
            return 0;
        }

        const suggestRect = suggest.getBoundingClientRect() || {};
        const {height = 0, top = 0} = suggestRect;

        if (this.wrapper.offsetHeight - SUGGEST_SHIFT - height - top >= 0) {
            return this.defaultWrapperHeight;
        }

        return height + top + SUGGEST_SHIFT;
    };

    onComponentsPropsUpdate = () => {
        const {retpath} = this.props;
        const heightWithError = this.wrapper.offsetHeight;
        const heightWithSuggest = this.getHeightWithSuggest();

        let formHeight = this.defaultWrapperHeight;

        if (heightWithError || heightWithSuggest) {
            formHeight = heightWithError > heightWithSuggest ? heightWithError : heightWithSuggest;
        }

        this.setState({formHeight});
        sendMessage({formHeight}, retpath);
    };

    render() {
        const {
            isTouch,
            isNewLayout,
            isPublicIdEnabled,
            displayName: {error, isProcessing}
        } = this.props;
        const isButtonDisabled = Boolean(error && isProcessing);

        return (
            <div
                className={classnames(WRAPPER_CLASSNAME, {
                    'passp-avatar_is-touch': isTouch,
                    'passp-avatar_nl': isNewLayout
                })}
            >
                <h1 data-t='title' className='passp-avatar_displayname__title'>
                    {i18n('Frontend.public_avatar-displayname.title')}
                </h1>
                <Avatar isTouch={isTouch} onPropsUpdate={this.onComponentsPropsUpdate} />

                <form method='post' action='/profile/public/displayname' onSubmit={this.handleSubmit}>
                    <DisplayName
                        isTouch={isTouch}
                        onPropsUpdate={this.onComponentsPropsUpdate}
                        suggestId={SUGGEST_ID}
                    />
                    {isPublicIdEnabled && <PublicId />}

                    <PersonalDataAllow />
                    <Button
                        type='submit'
                        view='action'
                        size='l'
                        width='max'
                        disabled={isButtonDisabled}
                        onClick={this.handleSubmit}
                        dataT='confirm'
                    >
                        {i18n('_AUTH_.common.save')}
                    </Button>
                </form>

                <PersonalDataAllowPopup
                    closePersonalDataAllowPopup={this.closePersonalDataAllowPopup}
                    handlePublicAccess={this.handlePublicAllowPopup}
                />
            </div>
        );
    }
}

export default connect(mapStateToProps, mapDispatchToProps)(User);

function mapStateToProps(state) {
    const {
        settings: {ua, isNewLayout},
        common: {retpath},
        publicId = {},
        displayName = {},
        publicAccess = {}
    } = state;
    const formData = getFormValues(state);
    const {isTouch, isMobile} = ua;
    const {isEnabled} = publicId;
    const {hasPublicProfile, hasThirdPartyAccess, isPopupVisible} = publicAccess;

    return {
        isTouch,
        isMobile,
        retpath,
        displayName,
        isNewLayout,
        publicId: formData.publicId,
        isPublicIdEnabled: isEnabled,
        hasPublicProfile,
        hasThirdPartyAccess,
        isPublicAccessPopupVisible: isPopupVisible,
        isPublicAllowCheckbox: hasExp(state, 'personal_data_checkbox_on')
    };
}

User.propTypes = {
    isNewLayout: PropTypes.bool,
    isTouch: PropTypes.bool.isRequired,
    retpath: PropTypes.string,
    isPublicIdEnabled: PropTypes.bool,
    hasPublicProfile: PropTypes.bool,
    hasThirdPartyAccess: PropTypes.bool,
    displayName: PropTypes.object,
    handleUpdates: PropTypes.func.isRequired,
    showPublicAccessPopup: PropTypes.func.isRequired,
    isPublicAccessPopupVisible: PropTypes.bool,
    isPublicAllowCheckbox: PropTypes.bool,
    toggleProfilePublicAccess: PropTypes.func.isRequired,
    toggleProfileThirdPartyAccess: PropTypes.func.isRequired,
    togglePersonalDataPermission: PropTypes.func.isRequired
};
