import cropperjs from '../../../../node_modules/cropperjs/dist/cropper.min.js';
import handleError from '@blocks/avatar-displayname/utils/setErrors';
import dataURItoBlob from '@blocks/avatar-displayname/utils/dataUriToBlob';
import {
    changeUpdateStatus,
    setAvatarProcessingStatus,
    setLoadingStatus,
    updateAvatarUrl
} from '@blocks/avatar-displayname/actions';
import api from '@blocks/api';
import {DEFAULT_AVATAR_ID, DEFAULT_AVATAR_SIZE} from '@blocks/avatar-displayname/avatarConsts';
import {updateHeaderAvatar} from '@blocks/morda/components/header/actions';
import {sendMessage} from '@blocks/utils';
import sendMetrics from '@blocks/avatar-displayname/utils/sendMetrics';
import setDisplayName from '@blocks/avatar-displayname/actions/setDisplayName';

export const avatarCropper = (() => {
    let cropperInst = null;

    return {
        init() {
            setCropper();
        },

        cropAvatar(shouldSetDisplayName = false) {
            return (dispatch, getState) => {
                const avatarState = getState().avatar;
                const result = cropperInst.getCroppedCanvas({fillColor: 'white'});
                const dataURL = result && result.toDataURL(`image/${avatarState.fileType}`, avatarState.compression);

                if (dataURL) {
                    dispatch(loadCroppedAvatar(dataURL, shouldSetDisplayName));
                    cropperInst.destroy();
                    return;
                }

                dispatch(handleError('error-internal'));
            };
        },

        destroyCropper() {
            cropperInst.destroy();
            cropperInst = null;
        }
    };

    function setCropper() {
        const image = document.getElementById('user-avatar');
        const Cropper = cropperjs;

        if (cropperInst) {
            cropperInst.destroy();
        }

        cropperInst = new Cropper(image, {
            aspectRatio: 1 / 1, // https://github.com/fengyuanchen/cropperjs#aspectratio
            checkCrossOrigin: false,
            guides: false,
            center: false,
            background: false,
            zoomable: false,
            movable: false,
            minContainerWidth: 120
        });

        image.addEventListener('crop', setMinCropperWidth);
        image.addEventListener('cropend', setMinCropperWidth);
    }

    function setMinCropperWidth() {
        const frameParams = cropperInst.getData(true);
        const frameWidth = frameParams.width;
        const minWidth = 200;

        if (frameWidth < minWidth) {
            cropperInst.setData({width: minWidth});
        }
    }
})();

export const loadCroppedAvatar = (croppedData, shouldSetDisplayName) => {
    return (dispatch, getState) => {
        const formData = new FormData();
        const file = dataURItoBlob(croppedData);
        const state = getState();
        const {
            avatar: {default_300: defaultAvatar, track_id: trackId, isOptionalStep},
            common: {csrf, retpath, source, mode},
            auth = {}
        } = state;
        const authMode = auth.mode;
        const options = {
            processData: false,
            contentType: false
        };
        const statboxMode = authMode === 'addAvatar' ? 'domik_ask_avatar' : source || mode;

        formData.append('file', file, 'avatar.jpg');
        formData.append('track_id', trackId);
        formData.append('csrf_token', csrf);

        dispatch(setLoadingStatus(true));

        api.request('/profile/avatars/load/', formData, options)
            .done((response) => {
                const {body = {}} = response;

                if (body && body.status === 'ok') {
                    const urlSplitted = body.avatar_url && body.avatar_url.split('/');
                    const avatarId = `${urlSplitted[2]}/${urlSplitted[3]}` || DEFAULT_AVATAR_ID;

                    dispatch(
                        updateAvatarUrl({
                            url: body.avatar_url ? `//${body.avatar_url}${DEFAULT_AVATAR_SIZE}` : defaultAvatar
                        })
                    );
                    dispatch(updateHeaderAvatar(avatarId));
                    sendMessage({updateAvatar: 'success'}, retpath);
                    sendMetrics('Загрузить аватар', 'avatar-displayname_load_avatar__success');

                    if (isOptionalStep) {
                        dispatch(changeUpdateStatus(true));
                    }

                    if (shouldSetDisplayName) {
                        dispatch(setDisplayName());
                    }

                    api.writeStatbox({
                        mode: statboxMode,
                        track_id: trackId,
                        action: 'update'
                    });
                } else {
                    dispatch(handleError('error-internal'));
                }
            })
            .fail((error) => {
                dispatch(handleError(error.errors));
                sendMetrics('Загрузить аватар', 'avatar-displayname_load_avatar__error');
            })
            .always(() => {
                dispatch(setAvatarProcessingStatus(false));
                dispatch(setLoadingStatus(false));
                avatarCropper.destroyCropper();
            });
    };
};
