import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {Input} from '@components/Input';
import {Popup} from '@components/Popup';
import * as extracted from './current_password.js';

export default class CurrentPassword extends React.PureComponent {
    passwordRef = React.createRef();

    constructor(props) {
        super(props);

        this.state = {
            passwordHidden: true
        };
        this.toggleFieldType = extracted.toggleFieldType.bind(this);
        this.updateAndCheck = extracted.updateAndCheck.bind(this);
        this.checkIfInvalid = extracted.checkIfInvalidHandler.bind(this);
    }

    render() {
        const passwordHidden = this.state.passwordHidden;
        const field = 'password';
        const fieldType = passwordHidden ? 'password' : 'text';
        const {value, state, error, active, isMobile, isAutoComplete} = this.props;
        const invalid = state === 'not_valid';
        const fieldTogglerText = passwordHidden
            ? i18n('Frontend.password_short_view')
            : i18n('Frontend.password_short_hide');

        return (
            <div
                className={classnames('form__field field-wrapper field__password', {
                    form__field_filled: value,
                    field__error: invalid
                })}
            >
                {isAutoComplete === false && (
                    <div className='registration__chrome_bag_autocomplete'>
                        <input type={fieldType} name='current_password' tabIndex='-1' />
                    </div>
                )}
                <button
                    className={!isMobile ? `field-type__toggler field-type-${fieldType}` : 'type__toggler field__btn'}
                    onClick={this.toggleFieldType}
                    title={fieldTogglerText}
                    tabIndex='1'
                    type='button'
                >
                    {fieldTogglerText}
                </button>
                <Input
                    id={field}
                    value={value}
                    name='current_password'
                    type={fieldType}
                    controlRef={this.passwordRef}
                    autoComplete={isAutoComplete === false ? 'off' : 'current-password'}
                    tabIndex='0'
                    onFocus={this.checkIfInvalid}
                    onChange={this.updateAndCheck}
                    view='one-border'
                    filled={Boolean(value)}
                />
                <label htmlFor={field} className='registration__label'>
                    {i18n('_AUTH_.field_password_current')}
                </label>
                {!isMobile && (
                    <Popup
                        visible={active && invalid}
                        hasTail={true}
                        directions={['right-center']}
                        anchor={this.passwordRef}
                        target='anchor'
                    >
                        <div className='form__popup-error' data-t='complete:current_password_popup-error'>
                            <div className='error-message'>{error.text}</div>
                            {error.descriptionText && <div className='error-desc'>{error.descriptionText}</div>}
                        </div>
                    </Popup>
                )}
                {isMobile && invalid && (
                    <div data-t='complete:current_pass-error'>
                        {error.text && <div className='form__field-error error-message'>{error.text}</div>}
                        {error.descriptionText && (
                            <div className='suggest__description-text'>{error.descriptionText}</div>
                        )}
                    </div>
                )}
            </div>
        );
    }
}

CurrentPassword.propTypes = {
    value: PropTypes.string.isRequired,
    error: PropTypes.object.isRequired,
    state: PropTypes.string.isRequired,
    active: PropTypes.bool.isRequired,
    dispatch: PropTypes.func.isRequired,
    isMobile: PropTypes.bool,
    isAutoComplete: PropTypes.bool
};
