import React from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';

import Promo from './promo.jsx';
import Form from './form.jsx';
import ConnectMobileHeader from './mobile/header.jsx';
import MobileFooter from '../registration/mobile/MobileFooter.jsx';
import RegistrationTitle from '../registration/fieldComponents/RegistrationTitle.jsx';

const texts = {
    tracker: {
        title: i18n('Frontend.promo.connect.title'),
        desc: i18n('Frontend.workspace.tracker_about')
    },
    connect: {
        title: i18n('Frontend.promo.connect.title'),
        desc: i18n('Frontend.promo.connect.desc')
    },
    forms: {
        title: i18n('Frontend.promo.connect.title'),
        desc: i18n('Frontend.promo.forms.descr')
    }
};
const srcs = {
    yandex: {
        ru: '//yastatic.net/q/logoaas/v1/%D0%AF%D0%BD%D0%B4%D0%B5%D0%BA%D1%81.svg',
        en: '//yastatic.net/q/logoaas/v1/Yandex.svg'
    },
    connect: {
        ru: '//yastatic.net/q/logoaas/v1/%D0%9A%D0%BE%D0%BD%D0%BD%D0%B5%D0%BA%D1%82.svg',
        en: '//yastatic.net/q/logoaas/v1/Connect.svg'
    },
    tracker: {
        ru: '//yastatic.net/q/logoaas/v1/Трекер.svg',
        en: '//yastatic.net/q/logoaas/v1/Tracker.svg'
    },
    forms: {
        ru: '//yastatic.net/q/logoaas/v1/Формы.svg',
        en: '//yastatic.net/q/logoaas/v1/Forms.svg'
    }
};

const originsWithCustomization = ['tracker', 'forms'];

class Connect extends React.PureComponent {
    static mapStateToProps(state) {
        const {
            common,
            settings,
            registrationErrors,
            footer,
            fetchingStatus: {isFetching},
            form
        } = state;

        return {
            footer,
            common,
            settings,
            registrationErrors,
            isFetching,
            form
        };
    }

    constructor(props) {
        super(props);

        this.defineOrigin = this.defineOrigin.bind(this);
        this.defineServices = this.defineServices.bind(this);
    }

    defineServices() {
        const services = ['yamb', 'wiki', 'connect', 'mail', 'disk'];
        const origin = this.defineOrigin();

        if (originsWithCustomization.includes(origin)) {
            const indexToChangeValue = services.findIndex((service) => service === 'connect');

            services[indexToChangeValue] = origin;
        }
        return services;
    }

    defineOrigin() {
        let {origin} = this.props.settings;

        if (origin !== 'connect' && !originsWithCustomization.includes(origin)) {
            origin = 'connect';
        }
        return origin;
    }

    render() {
        const {
            registrationErrors,
            footer: {langlist},
            settings,
            form,
            dispatch,
            isFetching
        } = this.props;
        const {links, language, isMobile, env, help, tld} = settings;
        const logoLang = ['en', 'fr', 'tr', 'fi', 'id'].includes(language) ? 'en' : 'ru';
        const origin = this.defineOrigin();
        const serviceLink = origin === 'tracker' ? links.tracker : links.portal;
        const servicesList = this.defineServices();
        const logos = {
            yandex: srcs.yandex[logoLang],
            service: srcs[origin][logoLang]
        };
        const footer = {langlist, language, env, help, tld};

        return (
            <div
                className={classnames({
                    'main-container': !isMobile,
                    'mobile_unified-experiment is_mobile_reg': isMobile
                })}
            >
                <ConnectMobileHeader
                    isMobile={isMobile}
                    links={links}
                    logos={logos}
                    serviceLink={serviceLink}
                    origin={origin}
                    texts={texts}
                    servicesList={servicesList}
                />
                {!isMobile && (
                    <Promo
                        registrationErrors={registrationErrors}
                        links={{yandex: links.yandex, service: serviceLink}}
                        service={{
                            name: origin,
                            list: servicesList,
                            link: serviceLink
                        }}
                        texts={texts[origin]}
                        logos={logos}
                        footer={footer}
                    />
                )}
                <div className={classnames('registration__wrapper', {registration__wrapper_desktop: !isMobile})}>
                    <div
                        className={classnames({registration__block: !isMobile, 'registration__block-mobile': isMobile})}
                    >
                        <RegistrationTitle isMobile={isMobile} title={i18n('Frontend.registration.connect.title')} />
                        <Form
                            isMobile={isMobile}
                            dispatch={dispatch}
                            form={form}
                            registrationErrors={registrationErrors}
                            isFetching={isFetching}
                            service={origin}
                        />
                    </div>
                </div>
                <MobileFooter isFetching={isFetching} isMobile={isMobile} />
            </div>
        );
    }
}

Connect.propTypes = {
    registrationErrors: PropTypes.object.isRequired,
    dispatch: PropTypes.func.isRequired,
    footer: PropTypes.object.isRequired,
    settings: PropTypes.object.isRequired,
    isFetching: PropTypes.bool,
    form: PropTypes.object.isRequired
};

Connect.childContextTypes = {
    prefix: PropTypes.string
};

export default connect(Connect.mapStateToProps)(Connect);
