import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import debounce from 'lodash/debounce';

import {Input} from '@components/Input';
import {Popup} from '@components/Popup';

import * as extracted from './domain_input.js';

export default class DomainInput extends PureComponent {
    inputRef = React.createRef();

    constructor(props) {
        super(props);

        this.state = {
            focused: false,
            placeholderHidden: false
        };
        this.handleBlur = extracted.handleBlur.bind(this);
        this.handleInput = extracted.handleInput.bind(this);
        this.handleFocus = extracted.handleFocus.bind(this);
        this.checkErrorVisibility = extracted.checkErrorVisibility.bind(this);
        this.handleValidation = debounce(extracted.handleValidation.bind(this), 600);

        extracted.construct.call(this, props);

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.state === 'need_validation') {
            this.handleValidation(this.props.value);
        }

        this.forceUpdate();
    }

    /* eslint-disable no-extra-parens */
    render() {
        const {state, value: domain, error, isMobile, service} = this.props;
        const invalid = state === 'not_valid';
        const valid = state === 'valid';
        const texts = {
            tracker: i18n('Frontend.hint.workspace.desc'),
            connect: i18n('Frontend.hint.workspace.connect.desc')
        };
        const hintText = texts[service] || texts.connect;
        const stopNumbers = this.stopNumbers;
        const placeholderHidden = (!this.state.focused && !domain.length) || domain.length > stopNumbers.main;
        const errorShown = this.checkErrorVisibility();

        return (
            <div
                className={classnames('form__field form__field_wide', {
                    'field-wrapper': isMobile,
                    form__field_filled: domain.length > 0 || this.isLoading,
                    field__error: invalid,
                    field__valid: valid
                })}
            >
                <Input
                    id='domain'
                    name='domain'
                    value={domain}
                    controlRef={this.inputRef}
                    onChange={this.handleInput}
                    onFocus={this.handleFocus}
                    onBlur={this.handleBlur}
                    view='one-border'
                    state={invalid ? 'error' : ''}
                    filled={domain.length > 0 || this.isLoading}
                />
                <label htmlFor='domain' className='registration__label'>
                    {i18n('Frontend.field.workspace.domain')}
                </label>
                {!this.state.placeholderHidden && (
                    <div
                        className={classnames('placeholder', {
                            placeholder_hidden: placeholderHidden,
                            placeholder_pushed: valid
                        })}
                    >
                        <label htmlFor='domain'>.yaconnect.com</label>
                    </div>
                )}
                <div
                    className={classnames('placeholder__bottom', {
                        placeholder__bottom_hidden: !placeholderHidden
                    })}
                >
                    {(domain.length > 10 &&
                        `${domain.substr(0, stopNumbers.sub)}${domain.length > stopNumbers.sub ? '...' : ''}`) ||
                        ''}
                    .yaconnect.com
                </div>
                {!isMobile && (
                    <div>
                        <Popup
                            visible={errorShown}
                            hasTail={true}
                            directions={['right-center']}
                            anchor={this.inputRef}
                            target='anchor'
                        >
                            <div className='form__popup-error' data-t='connect:domain-error'>
                                <div className='error-message'>{error.text}</div>
                                {error.descriptionText && <div className='error-desc'>{error.descriptionText}</div>}
                            </div>
                        </Popup>
                        <Popup
                            visible={!errorShown && this.state.focused}
                            hasTail={true}
                            directions={['right-center']}
                            anchor={this.inputRef}
                            target='anchor'
                        >
                            <div className='form__popup-text'>
                                <div dangerouslySetInnerHTML={{__html: hintText}} />
                            </div>
                        </Popup>
                    </div>
                )}
                {isMobile &&
                    (({text, descriptionText}, showError) => {
                        if (showError) {
                            return (
                                <div
                                    className={classnames({placeholder__margin: placeholderHidden})}
                                    data-t='connect:error'
                                >
                                    {text && <div className='form__field-error error-message'>{text}</div>}
                                    {descriptionText && (
                                        <div className='suggest__description-text'>{descriptionText}</div>
                                    )}
                                </div>
                            );
                        }

                        return (
                            <div
                                className={classnames('form__field-hint', {
                                    placeholder__margin: placeholderHidden
                                })}
                                dangerouslySetInnerHTML={{__html: hintText}}
                            />
                        );
                    })(error, errorShown)}
            </div>
        );
    }
}

DomainInput.propTypes = {
    value: PropTypes.string.isRequired,
    state: PropTypes.string.isRequired,
    error: PropTypes.object.isRequired,
    active: PropTypes.bool.isRequired,
    dispatch: PropTypes.func.isRequired,
    isMobile: PropTypes.bool,
    service: PropTypes.string.isRequired
};
