import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import debounce from 'lodash/debounce';

import {Input} from '@components/Input';
import {Popup} from '@components/Popup';

import * as extracted from './login_input.js';

export default class LoginInput extends React.Component {
    inputRef = React.createRef();

    constructor(props) {
        super(props);

        this.onBlur = extracted.onBlur.bind(this);
        this.handleInput = extracted.handleInput.bind(this);
        this.handleFocus = extracted.handleFocus.bind(this);
        this.handleValidation = debounce(extracted.handleValidation.bind(this), 600);

        extracted.construct.call(this, props);

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.states.login === 'need_validation') {
            this.handleValidation(this.props.values.login);
        }

        this.forceUpdate();
    }

    shouldComponentUpdate(nextProps, nextStates) {
        return extracted.shouldComponentUpdate.call(this, nextProps, nextStates);
    }

    /* eslint-disable no-extra-parens */
    render() {
        const {
            states,
            values: {login, domain},
            errors,
            isMobile
        } = this.props;
        const {login: loginError} = errors;
        const invalid = states.login === 'not_valid';
        const valid = states.login === 'valid';
        const stopNumbers = this.stopNumbers;
        const domainLength = extracted.getDomainLength.call(this);
        const placeholderHidden = (!this.state.focused && !login.length) || extracted.isValueTooLong.call(this, login);
        const tail = (
            <span>
                @
                {(domain.length > 0 &&
                    states.domain === 'valid' &&
                    `${domain.substr(0, 10)}${domain.length >= 10 ? '...' : ''}`) || (
                    <span className='placeholder_empty'>name</span>
                )}
                .yaconnect.com
            </span>
        );

        return (
            <div
                className={classnames('form__field form__field_wide', {
                    'field-wrapper': isMobile,
                    form__field_filled: login.length > 0 || this.isLoading,
                    field__error: invalid,
                    field__valid: valid
                })}
            >
                <Input
                    id='login'
                    name='login'
                    value={login}
                    controlRef={this.inputRef}
                    onChange={this.handleInput}
                    onFocus={this.handleFocus}
                    onBlur={this.onBlur}
                    view='one-border'
                    state={invalid ? 'error' : ''}
                    filled={login.length > 0 || this.isLoading}
                />
                <label htmlFor='login' className='registration__label'>
                    {i18n('_AUTH_.field_login')}
                </label>
                {!this.state.placeholderHidden && (
                    <div
                        className={classnames('placeholder', {
                            placeholder_hidden: placeholderHidden,
                            placeholder_pushed: valid
                        })}
                    >
                        <label htmlFor='login'>{tail}</label>
                    </div>
                )}
                <div
                    className={classnames('placeholder__bottom', {
                        placeholder__bottom_hidden: !placeholderHidden
                    })}
                >
                    {login.substr(0, 22 - domainLength - stopNumbers.sub)}
                    {22 - domainLength - stopNumbers.sub < login.length ? '...' : ''}
                    {tail}
                </div>
                {!isMobile && (
                    <Popup
                        target='anchor'
                        visible={invalid && errors.active === 'login' && !errors.hintActive}
                        hasTail={true}
                        directions={['right-center']}
                        anchor={this.inputRef}
                    >
                        <div className='form__popup-error' data-t='connect:login-error'>
                            <div className='error-message'>{loginError.text}</div>
                            {loginError.descriptionText && (
                                <div className='error-desc'>{loginError.descriptionText}</div>
                            )}
                        </div>
                    </Popup>
                )}
                {isMobile && invalid && (
                    <div className={classnames({placeholder__margin: placeholderHidden})} data-t='connect:login-error'>
                        {loginError.text && <div className='form__field-error error-message'>{loginError.text}</div>}
                        {loginError.descriptionText && (
                            <div className='suggest__description-text'>{loginError.descriptionText}</div>
                        )}
                    </div>
                )}
            </div>
        );
    }
}

LoginInput.propTypes = {
    values: PropTypes.object.isRequired,
    states: PropTypes.object.isRequired,
    errors: PropTypes.object.isRequired,
    dispatch: PropTypes.func.isRequired,
    isMobile: PropTypes.bool
};
