import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Modal} from '@components/Modal';
import DeleteForm from './deleteForm.jsx';
import DeleteWarning from './deleteWarning.jsx';
import {BackButton} from '@components/BackButton';
import {redirectToRetpathFromWebview} from '@blocks/utils';
import {WEBVIEW_STATUS_CANCEL} from '@blocks/constants';

export default class DeleteComponent extends PureComponent {
    constructor(props) {
        super(props);
        this.returnMessageText = this.returnMessageText.bind(this);
        this.returnPopupMessageText = this.returnPopupMessageText.bind(this);
        this.renderSocialchik = this.renderSocialchik.bind(this);
        this.renderLastDeleteWarning = this.renderLastDeleteWarning.bind(this);
        this.renderModal = this.renderModal.bind(this);
    }

    closeWebview() {
        redirectToRetpathFromWebview(null, WEBVIEW_STATUS_CANCEL);
    }

    renderSocialchik() {
        const {isAppMode} = this.props;
        const upgradeLink = `/profile/upgrade?retpath=${this.props.currentPage}${isAppMode ? '&mode=app' : ''}`;
        const socialUserText = i18n('ProfileDelete.is-social-user_text').replace('%', upgradeLink);

        return (
            <div
                data-t='description'
                className='delete-description social-user__text'
                dangerouslySetInnerHTML={{__html: socialUserText}}
            />
        );
    }

    static renderPddAdmin() {
        return (
            <div data-t='description' className='delete-description pdd-admin__text'>
                {i18n('ProfileDelete.is-pdd-user_text')}
            </div>
        );
    }

    returnMessageText() {
        const {hasBlockingSids, hasMailSid} = this.props.deleteAccount;
        const warningTexts = {
            blockingSids: i18n('ProfileDelete.popup_delete-blocking-sid-text'),
            noMailSid: i18n('ProfileDelete.popup_delete-nomail-text'),
            normal: i18n('ProfileDelete.popup_delete-text')
        };

        let text = warningTexts.normal;

        if (hasBlockingSids) {
            text = warningTexts.blockingSids;
        }

        if (!hasMailSid) {
            text = warningTexts.noMailSid;
        }

        return text;
    }

    returnPopupMessageText() {
        const {hasMailSid} = this.props.deleteAccount;

        if (hasMailSid) {
            return i18n('ProfileDelete.popup_mail_updated');
        }
        return i18n('ProfileDelete.popup_nomail_updated');
    }

    renderLastDeleteWarning() {
        const {isMobile, isTablet, dispatch, isAppMode} = this.props;
        const popUpText = this.returnPopupMessageText();

        return (
            <DeleteWarning text={popUpText} isMobile={isMobile || isTablet} dispatch={dispatch} isAppMode={isAppMode} />
        );
    }

    renderModal() {
        const {isMobile} = this.props;
        const {isModalOpened} = this.props.deleteAccount;

        if (isMobile) {
            return null;
        }

        return <Modal visible={isModalOpened}>{isModalOpened && this.renderLastDeleteWarning()}</Modal>;
    }

    render() {
        const {isModalOpened, isSocialchik, isPddAdmin} = this.props.deleteAccount;
        const {isMobile, isTablet, isAppMode} = this.props;
        const isNormalUser = !isSocialchik && !isPddAdmin;
        const isShowLastWarningOnMobiles = (isModalOpened && isMobile) || (isModalOpened && isTablet);
        const pageDescriptionText = this.returnMessageText();

        if (isShowLastWarningOnMobiles) {
            return this.renderLastDeleteWarning();
        }

        return (
            <div className='delete__user-account'>
                <div className='section-head'>
                    {isAppMode && (
                        <div className='return-button-wrapper'>
                            <BackButton onClick={this.closeWebview} mode='arrowBack' place='newLayout' />
                        </div>
                    )}
                    <h2 data-t='title' className='section-title'>
                        {i18n('ProfileDelete.delete.main-title')}
                    </h2>

                    {isSocialchik && this.renderSocialchik()}

                    {isPddAdmin && DeleteComponent.renderPddAdmin()}

                    {isNormalUser && <DeleteForm text={pageDescriptionText} {...this.props} />}
                </div>
                {this.renderModal()}
            </div>
        );
    }
}

DeleteComponent.propTypes = {
    deleteAccount: PropTypes.object.isRequired,
    currentPage: PropTypes.string,
    common: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    isMobile: PropTypes.bool.isRequired,
    isTablet: PropTypes.bool.isRequired,
    isAppMode: PropTypes.bool
};
