import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {Input} from '@components/Input';
import {Spin} from '@components/Spin';
import updateFormFields from '../helpers/update_form_fields';
import processError from '../helpers/process_error';
import initDeleteProcess from '../helpers/init_delete_process';
import deleteAccountCommit from '../thunks/delete_account_commit';

import WrappedCaptchaContainer from '../captchaContainer.jsx';
import PhoneCheck from './phoneCheck.jsx';
import QuestionCheck from './questionCheck.jsx';
import EmailCheck from './emailCheck.jsx';

const COMPONENTS_LIST = {
    phone: PhoneCheck,
    question: QuestionCheck,
    email: EmailCheck
};

export const ConfirmationComponent = (props) => {
    const confirmationMethod = props.confirmation.method;

    if (confirmationMethod === 'captcha') {
        return null;
    }

    const Handler = COMPONENTS_LIST[confirmationMethod];

    return <Handler {...props} />;
};

ConfirmationComponent.propTypes = {
    confirmation: PropTypes.object.isRequired
};

export const FormButtons = ({isFetching, isAppMode}) => (
    <div className='delete-account__buttons'>
        <div className='fetching-block'>
            <Spin progress={isFetching} size='m' />
        </div>
        <Button
            dataT='cancel-delete'
            size='l'
            view='pseudo'
            type='link'
            url={isAppMode ? '/closewebview?status=close' : '/profile'}
        >
            <span
                dangerouslySetInnerHTML={
                    isAppMode ? {__html: i18n('Mend.back_to_app')} : {__html: i18n('Mend.back_to_passp')}
                }
            />
        </Button>
        <Button dataT='confirm-delete' view='action' size='l' type='submit'>
            <span dangerouslySetInnerHTML={{__html: i18n('Profile.personal.hangover.remove')}} />
        </Button>
    </div>
);

FormButtons.propTypes = {
    isFetching: PropTypes.bool,
    isAppMode: PropTypes.bool
};

export default class DeletionChecks extends PureComponent {
    constructor(props) {
        super(props);
        this.deleteAccount = this.deleteAccount.bind(this);
        this.onInput = this.onInput.bind(this);
        this.onSubmit = this.onSubmit.bind(this);
        this.checkStatus = this.checkStatus.bind(this);
        this.defineFormTitleText = this.defineFormTitleText.bind(this);
    }

    onInput(event) {
        const {target = {}} = event;
        const {value, name} = target;
        const fieldInfo = {};

        fieldInfo[name] = value;
        this.props.dispatch(updateFormFields(fieldInfo));
    }

    onSubmit(event) {
        const {status} = this.props.confirmation;
        const isAccountNotConfirmed = status !== 'confirmed' && status !== 'code_checked';

        event.preventDefault();
        if (isAccountNotConfirmed) {
            this.checkStatus();
        }
    }

    checkStatus() {
        const {form, captchaPassed, dispatch} = this.props;

        if (!form.captcha && !captchaPassed) {
            dispatch(processError('captcha', 'captcha.missingvalue'));
            return;
        }

        dispatch(initDeleteProcess());
    }

    deleteAccount() {
        this.props.dispatch(deleteAccountCommit());
    }

    defineFormTitleText() {
        const confirmationMethod = (this.props.confirmation || {}).method;
        const texts = {
            question: i18n('ProfileDelete.form_question-title'),
            phone: i18n('ProfileDelete.form_phone-title'),
            email: i18n('ProfileDelete.form_email-title')
        };

        return texts[confirmationMethod];
    }

    render() {
        const {confirmation, error, form, dispatch, isFetching, isMobile, isAppMode} = this.props;

        return (
            <div className='delete-account_checks'>
                <h3 className='delete__service-list__title'>{this.defineFormTitleText()}</h3>
                <form onSubmit={this.onSubmit}>
                    {error.field === 'global' && (
                        <div className='errors-block input-error error-block_global'>{error.message}</div>
                    )}

                    {confirmation.status === 'unconfirmed' && (
                        <div className='delete-account_captcha delete-account__fields'>
                            <div className='delete-account__control-label'>{i18n('Frontend.field_answer')}:</div>
                            <WrappedCaptchaContainer />
                            <div className='delete-account__control-input'>
                                <Input
                                    size='l'
                                    id='captcha'
                                    value={form.captcha}
                                    hasClear={true}
                                    placeholder={i18n('ProfileDelete.input_captcha-placeholder')}
                                    name='captcha'
                                    onChange={this.onInput}
                                />
                                <div className='errors-block input-error error-block_captcha'>
                                    {error.field === 'captcha' && error.message}
                                </div>
                            </div>
                        </div>
                    )}
                    <ConfirmationComponent
                        confirmation={confirmation}
                        form={form}
                        error={error}
                        isMobile={isMobile}
                        checkStatus={this.checkStatus}
                        dispatch={dispatch}
                    />
                    <FormButtons isFetching={isFetching} isAppMode={isAppMode} />
                </form>
            </div>
        );
    }
}

DeletionChecks.propTypes = {
    confirmation: PropTypes.object.isRequired,
    captchaPassed: PropTypes.bool,
    error: PropTypes.object,
    form: PropTypes.object.isRequired,
    dispatch: PropTypes.func.isRequired,
    isFetching: PropTypes.bool,
    isMobile: PropTypes.bool,
    isAppMode: PropTypes.bool
};
