import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Input} from '@components/Input';
import {Button} from '@components/Button';
import {Link} from '@components/Link';
import {Select} from '@components/Select';
import sendConfirmation from '../thunks/send_confirmation';
import processError from '../helpers/process_error';
import updateFormFields from '../helpers/update_form_fields';
import {checkStatus} from '../actions';
import InputError from './inputError.jsx';

class EmailCheck extends PureComponent {
    constructor(props) {
        super(props);
        this.sendCode = this.sendCode.bind(this);
        this.emailChange = this.emailChange.bind(this);
        this.onInputCode = this.onInputCode.bind(this);
        this.changeAddress = this.changeAddress.bind(this);
        this.sendCodeAgain = this.sendCodeAgain.bind(this);
        this.renderEmail = this.renderEmail.bind(this);
    }

    sendCode() {
        const {form, confirmation, dispatch} = this.props;

        if (!form.captcha && confirmation.status !== 'skip_captcha') {
            dispatch(processError('captcha', 'captcha.missingvalue'));
            return;
        }

        if (!form.email) {
            dispatch(processError('email', 'email.missingvalue'));
            return;
        }
        dispatch(sendConfirmation(form, 'email'));
    }

    sendCodeAgain() {
        this.props.dispatch(checkStatus('skip_captcha'));
        this.sendCode();
    }

    defineEmailList() {
        const {confirmation} = this.props;
        const emailAdresses = confirmation.emails;

        return emailAdresses.map((email, index) => ({
            content: email.text,
            value: email.value,
            index: index + 1
        }));
    }

    emailChange(event) {
        const {dispatch} = this.props;

        dispatch(updateFormFields({email: event.target.value}));
    }

    onInputCode(event) {
        const {target = {}} = event;
        const {value, name} = target;
        const fieldInfo = {};

        fieldInfo[name] = value;
        this.props.dispatch(updateFormFields(fieldInfo));
    }

    changeAddress() {
        const {dispatch} = this.props;

        dispatch(updateFormFields({}));
        dispatch(checkStatus('skip_captcha'));
    }

    renderEmail() {
        const {form, confirmation} = this.props;
        const {emails, status} = confirmation;
        const selectOptions = this.defineEmailList();
        const formattedEmail = selectOptions.find((el) => el.value === form.email).content;

        if (!emails) {
            return null;
        }

        if (emails.length === 1 || status === 'code_sent') {
            return (
                <div className='delete-account_email p-control-input__with-margin'>
                    <strong className='user-email' title={formattedEmail}>
                        {formattedEmail}
                    </strong>
                </div>
            );
        }

        return (
            <div className='delete-account_email-field'>
                <Select
                    width='max'
                    size='l'
                    value={form.email}
                    onChange={this.emailChange}
                    options={selectOptions}
                    name='email'
                    dataT='email'
                />
            </div>
        );
    }

    render() {
        const {error, form, confirmation, checkStatus} = this.props;
        const {emails, status} = confirmation;
        const sendCodeText =
            form.captcha === 'skip'
                ? i18n('ProfileDelete.notify.email-code_sent-again')
                : i18n('ProfileDelete.notify.email-code_sent');
        const hasCodeSent = status === 'code_sent';
        const isCheckCodeInputShown = hasCodeSent || status === 'code_checked';
        const isEmailErrorShowing = error.field === 'email';

        return (
            <div className='delete-account_email'>
                <div className='delete-account__fields'>
                    <div className='delete-account__control-label'>{i18n('ProfileDelete.email-field_label')}</div>
                    {this.renderEmail()}
                    <div className='delete-account_field-item delete-account_send-btn-field'>
                        <Button
                            view='action'
                            size='l'
                            type='button'
                            disabled={hasCodeSent}
                            onClick={this.sendCode}
                            dataT='send-code'
                            width='max'
                        >
                            {hasCodeSent ? sendCodeText : i18n('ProfileDelete.button_send-code')}
                        </Button>
                        {isEmailErrorShowing && (
                            <div className='errors-block input-error error-block_code'>{error.message}</div>
                        )}
                    </div>
                </div>
                {isCheckCodeInputShown && (
                    <div className='delete-account__fields'>
                        <div className='delete-account__control-label'>{i18n('ProfileDelete.code-field_label')}</div>
                        <div className='delete-account__control-input p-control-input__with-margin'>
                            <Input
                                size='l'
                                id='code'
                                value={form.code}
                                hasClear={true}
                                type='tel'
                                placeholder={i18n('ProfileDelete.code-field_email-placeholder')}
                                name='code'
                                onChange={this.onInputCode}
                            />
                            <InputError error={error} />
                        </div>
                        <div className='delete-account_field-item delete-account_send-btn-field'>
                            <Button
                                view='action'
                                size='l'
                                type='button'
                                onClick={checkStatus}
                                dataT='check-code'
                                width='max'
                            >
                                {i18n('_AUTH_.confirm_code')}
                            </Button>
                            <span className='field_links'>
                                {emails.length > 1 && (
                                    <Link pseudo={true} onClick={this.changeAddress}>
                                        {i18n('Profile.emails.change-email')}
                                    </Link>
                                )}
                                <Link pseudo={true} onClick={this.sendCodeAgain}>
                                    {i18n('_AUTH_.code.send-again-link')}
                                </Link>
                            </span>
                        </div>
                    </div>
                )}
            </div>
        );
    }
}

export default EmailCheck;

EmailCheck.propTypes = {
    confirmation: PropTypes.shape({
        method: PropTypes.string,
        status: PropTypes.string,
        emails: PropTypes.array
    }).isRequired,
    form: PropTypes.object,
    error: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    checkStatus: PropTypes.func.isRequired
};
