import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Input} from '@components/Input';
import {Button} from '@components/Button';
import {Link} from '@components/Link';
import sendConfirmation from '../thunks/send_confirmation';
import updateFormFields from '../helpers/update_form_fields';
import {checkStatus} from '../actions';
import checkCode from '../thunks/check_code';
import {getCountDownStarted, getCountDownStopped} from '../../common/phone_countdown/reducers';
import CountDown from '../../common/phone_countdown/CountDown.jsx';
import InputError from './inputError.jsx';

class PhoneCheck extends Component {
    constructor(props) {
        super(props);

        this.state = {
            hasCodeSent: false
        };

        this.sendCode = this.sendCode.bind(this);
        this.sendCodeAgain = this.sendCodeAgain.bind(this);
        this.onInputCode = this.onInputCode.bind(this);
        this.sendCodeToCheck = this.sendCodeToCheck.bind(this);
    }

    static mapStateToProps(state) {
        return {
            hasCountDownStarted: getCountDownStarted(state),
            hasCountDownStopped: getCountDownStopped(state)
        };
    }

    sendCode() {
        const {form, dispatch, hasCountDownStarted} = this.props;
        const {hasCodeSent} = this.state;

        if (hasCodeSent) {
            dispatch(checkStatus('skip_captcha'));
        }

        dispatch(sendConfirmation(form, 'phone'));

        if (hasCountDownStarted) {
            this.setState({
                hasCodeSent: true
            });
        }
    }

    sendCodeAgain() {
        this.props.dispatch(checkStatus('skip_captcha'));
        this.sendCode();
    }

    onInputCode(event) {
        const {target = {}} = event;
        const {value, name} = target;
        const fieldInfo = {};

        fieldInfo[name] = value;
        this.props.dispatch(updateFormFields(fieldInfo));
    }

    sendCodeToCheck() {
        const {form, dispatch} = this.props;

        dispatch(checkCode(form.code, 'phone'));
    }

    render() {
        const {form, error, confirmation, hasCountDownStarted, hasCountDownStopped} = this.props;
        const {phone, status} = confirmation;
        const hasCodeSent = status === 'code_sent';
        const buttonText = hasCodeSent
            ? i18n('ProfileDelete.notify.email-code_sent')
            : i18n('ProfileDelete.button_send-code');
        const isCheckCodeInputShown = hasCodeSent || status === 'code_checked';

        return (
            <div>
                <div className='delete-account__fields'>
                    <div className='delete-account__control-label'>
                        {i18n('ProfileJournal.profile.journal.personal-phone')}:
                    </div>
                    <div className='delete-account_phone-number-field'>
                        <strong className='delete-account__masked-phone'>{phone}</strong>
                    </div>
                    <div className='delete-account_field-item delete-account_send-btn-field'>
                        <Button
                            view='action'
                            size='l'
                            type='button'
                            onClick={this.sendCode}
                            dataT='send-code'
                            width='max'
                            disabled={hasCodeSent}
                        >
                            {buttonText}
                        </Button>
                        <div className='errors-block input-error error-block_phone'>
                            {error.field === 'phone' && error.message}
                        </div>
                    </div>
                </div>
                {isCheckCodeInputShown && (
                    <div className='delete-account__fields'>
                        <div className='delete-account__control-label'>{i18n('ProfileDelete.code-field_label')}</div>
                        <div className='delete-account__control-input p-control-input__with-margin'>
                            <Input
                                size='l'
                                id='code'
                                value={form.code}
                                hasClear={true}
                                type='tel'
                                placeholder={i18n('ProfileDelete.code-field_phone-placeholder')}
                                name='code'
                                onChange={this.onInputCode}
                            />
                            <InputError error={error} />
                        </div>
                        <div className='delete-account_field-item delete-account_send-btn-field'>
                            <Button
                                view='action'
                                size='l'
                                type='button'
                                onClick={this.sendCodeToCheck}
                                dataT='check-code'
                                width='max'
                            >
                                {i18n('_AUTH_.confirm_code')}
                            </Button>
                            <span className='field_links'>
                                {hasCountDownStarted && (
                                    <span className='link__disabled'>{i18n('_AUTH_.code.send-again-link')}</span>
                                )}
                                {hasCountDownStopped && (
                                    <Link pseudo={true} onClick={this.sendCodeAgain}>
                                        {i18n('_AUTH_.code.send-again-link')}
                                    </Link>
                                )}
                                <CountDown />
                            </span>
                        </div>
                    </div>
                )}
            </div>
        );
    }
}

export default connect(PhoneCheck.mapStateToProps)(PhoneCheck);

PhoneCheck.propTypes = {
    confirmation: PropTypes.shape({
        method: PropTypes.string,
        status: PropTypes.string,
        phone: PropTypes.string,
        countdown: PropTypes.number
    }).isRequired,
    form: PropTypes.object,
    error: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    hasCountDownStarted: PropTypes.bool,
    hasCountDownStopped: PropTypes.bool
};
