import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';

import {Input} from '@components/Input';
import {Select} from '@components/Select';

const ANOTHER = 'ANOTHER';

export default class EditDisplayName extends React.Component {
    constructor(props) {
        super(props);

        const person = props.person;

        if (person.displayNames) {
            this.options = Object.keys(person.displayNames).map((name) => ({
                content: name,
                value: person.displayNames[name]
            }));

            if (this.options.length) {
                this.options.push({
                    content: i18n('Profile.displayname.another'),
                    value: ANOTHER
                });
            }
        } else {
            this.options = [];
        }

        this.onSelect = this.onSelect.bind(this);
        this.onTextChage = this.onTextChage.bind(this);

        this.errors = {
            empty: i18n('_AUTH_.displayname_errors_missingvalue'),
            update_not_allowed: i18n('Frontend.displayname_errors_update_not_allowed'),
            invalid: i18n('_AUTH_.inputname.invalid'),
            long: i18n('_AUTH_.displayname_errors_toolong')
        };
    }

    onTextChage(event) {
        const {target = {}} = event;
        const {value} = target;

        this.props.onChange({name: value});
    }

    onSelect(event) {
        this.props.onChange({select: event.target.value});
    }

    render() {
        const {
            person: {errors = {}},
            select,
            name
        } = this.props;
        const isCustomDisplayName = this.options.length && select === ANOTHER;

        return (
            <div className='p-control-entry dname-edit__entry'>
                <div className='dname-edit__label'>{i18n('Profile.displayname.set')}:</div>
                {this.options.length && (
                    <div
                        className={classnames('p-control-entry-cntrl', {
                            'p-control-entry-cntrl__available_width': !isCustomDisplayName
                        })}
                    >
                        <Select
                            value={select}
                            size='l'
                            dataT='displayname'
                            name='displayname'
                            width='max'
                            onChange={this.onSelect}
                            options={this.options}
                        />
                    </div>
                )}
                <div
                    className={classnames('p-control-entry-cntrl', {
                        'g-hidden': !isCustomDisplayName,
                        'p-control-entry-cntrl__available_width': !this.options.length
                    })}
                >
                    <Input
                        size='l'
                        className='override'
                        type='text'
                        name='display_name'
                        value={name}
                        hasClear={true}
                        placeholder={i18n('Profile.displayname.label')}
                        onChange={this.onTextChage}
                    />
                </div>
                <span data-t='personal-info-displayname-error'>
                    {errors.display_name && (
                        <div className='p-control-error-block'>{this.errors[errors.display_name]}</div>
                    )}
                </span>
            </div>
        );
    }
}

EditDisplayName.propTypes = {
    person: PropTypes.shape({
        escapedDisplayName: PropTypes.string.isRequired,
        displayName: PropTypes.string.isRequired,
        displayNames: PropTypes.object,
        errors: PropTypes.object
    }).isRequired,
    dispatch: PropTypes.func.isRequired,
    onChange: PropTypes.func.isRequired,
    select: PropTypes.string.isRequired,
    name: PropTypes.string.isRequired
};
