import React from 'react';
import PropTypes from 'prop-types';

import {Link} from '@components/Link';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';

import ReturnToService from '@blocks/morda/components/return_button/return_button.jsx';
import AddressesField from '../addresses_field/index.jsx';

import {EDIT_MODE_DELIVERY_ADDRESSES, EDIT_MODE_ADDRESSES, DEFAULT_DELIVERY_ADDRESS_ID} from '../../actions/index';
import {checkGeoLocation} from '../../actions/check_geolocation';

import {
    closeAddressesControl,
    onSubmit,
    saveAddresses,
    setAddDeliveryAddressMode,
    handleAddressUpdate
} from './service';

class AddressesEditControl extends React.Component {
    constructor(props) {
        super(props);
        this.displayName = 'AddressesEditControl';

        this.closeAddressesControl = closeAddressesControl.bind(this);
        this.saveAddresses = saveAddresses.bind(this);
        this.setAddDeliveryAddressMode = setAddDeliveryAddressMode.bind(this);
        this.onSubmit = onSubmit.bind(this);
        this.handleAddressUpdate = handleAddressUpdate.bind(this);
    }

    componentDidMount() {
        const {dispatch} = this.props;

        dispatch(checkGeoLocation());
    }

    componentWillReceiveProps(nextProps) {
        if (!nextProps.addresses.isUpdated) {
            return;
        }

        this.handleAddressUpdate();
    }

    render() {
        const {
            addresses,
            dispatch,
            settings,
            modal,
            common: {edit, retpath}
        } = this.props;
        const {
            delivery = [],
            home = {
                id: 'home'
            },
            work = {
                id: 'work'
            },
            defaultDelivery = {
                id: DEFAULT_DELIVERY_ADDRESS_ID
            },
            mode = EDIT_MODE_ADDRESSES,
            suggest,
            context,
            editedFields = {},
            savingProgress = false,
            addDeliveryAddress = false,
            isGeoLocationAvailable = false,
            geoLocationUpdateState = {},
            editErrors = {}
        } = addresses;
        const isEdited = Object.keys(editedFields).length;
        const ua = settings.ua || {};
        const {isMobile} = ua;
        const isPhone = isMobile || (ua.isTouch && !ua.isTablet);

        let addrs = [home, work];

        const isValidForm = [home, work, defaultDelivery]
            .concat(delivery)
            .every((addr) => addr.isValid || !(addr.addressLine || '').trim());

        if (mode === EDIT_MODE_DELIVERY_ADDRESSES) {
            addrs = delivery;
        }

        if (edit !== EDIT_MODE_ADDRESSES && edit !== EDIT_MODE_DELIVERY_ADDRESSES && modal) {
            return null;
        }

        return (
            <form className='addresses-edit' onSubmit={this.onSubmit}>
                {addrs.map((address) => (
                    <AddressesField
                        key={address.id}
                        id={address.id}
                        context={context}
                        suggest={suggest}
                        dispatch={dispatch}
                        address={address}
                        isGeoLocationAvailable={isGeoLocationAvailable}
                        geoLocationUpdateState={geoLocationUpdateState}
                        errors={editErrors[address.id]}
                        isMobile={isPhone}
                    />
                ))}

                {mode === EDIT_MODE_DELIVERY_ADDRESSES && (
                    <div>
                        {!addrs.length || addDeliveryAddress ? (
                            <AddressesField
                                id={defaultDelivery.id}
                                key={defaultDelivery.id}
                                context={context}
                                address={defaultDelivery}
                                suggest={suggest}
                                dispatch={dispatch}
                                isGeoLocationAvailable={isGeoLocationAvailable}
                                geoLocationUpdateState={geoLocationUpdateState}
                                errors={editErrors[defaultDelivery.id]}
                                isMobile={isPhone}
                            />
                        ) : (
                            <Link pseudo={true} onClick={this.setAddDeliveryAddressMode}>
                                {i18n('Profile.addresses.add')}
                            </Link>
                        )}
                    </div>
                )}

                <div className='p-control-saveblock'>
                    <div className='p-control-saveblock-cell-right personal-saveblock'>
                        <span className='p-control-saveblock-button'>
                            <Button
                                view='action'
                                size='l'
                                type='submit'
                                disabled={Boolean(!isValidForm || !isEdited || savingProgress)}
                                onClick={this.saveAddresses}
                                text={i18n('Frontend.save')}
                                width={isMobile ? 'max' : 'auto'}
                            />
                        </span>
                        <span className='personal-spinner'>
                            <Spin progress={savingProgress} size='s' />
                        </span>
                    </div>

                    {modal ? (
                        <div className='p-control-saveblock-cell-left'>
                            <span className='p-control-saveblock-button'>
                                <Button
                                    view='pseudo'
                                    size='l'
                                    text={i18n('Frontend.close')}
                                    type='link'
                                    onClick={this.closeAddressesControl}
                                    width={isMobile ? 'max' : 'auto'}
                                />
                            </span>
                        </div>
                    ) : (
                        <ReturnToService retpath={retpath} isMobile={isPhone} isModal={false} />
                    )}
                </div>
            </form>
        );
    }
}

AddressesEditControl.propTypes = {
    dispatch: PropTypes.func.isRequired,
    addresses: PropTypes.shape({
        suggest: PropTypes.array,
        mode: PropTypes.string,
        context: PropTypes.string,
        editedFields: PropTypes.object,
        savingProgress: PropTypes.bool,
        editErrors: PropTypes.object,
        isUpdated: PropTypes.bool,
        addDeliveryAddress: PropTypes.bool,
        isGeoLocationAvailable: PropTypes.bool,
        delivery: PropTypes.array,
        home: PropTypes.object,
        work: PropTypes.object,
        defaultDelivery: PropTypes.object,
        geoLocationUpdateState: PropTypes.object
    }),
    common: PropTypes.shape({
        retpath: PropTypes.string,
        edit: PropTypes.string
    }),
    settings: PropTypes.object.isRequired,
    modal: PropTypes.bool
};

export default AddressesEditControl;
