import React from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import throttle from 'lodash/throttle';

import {Input} from '@components/Input';
import {Popup} from '@components/Popup';
import {Menu} from '@components/Menu';
import {Spin} from '@components/Spin';

import {ADDRESS_TYPES, DEFAULT_DELIVERY_ADDRESS_ID} from '../../actions';

import {
    ERRORS,
    getSuggest,
    setContext,
    clearContext,
    getOptions,
    isDeliveryAddress,
    getSuggestValue,
    deleteAddress,
    updatedDeliveryAddress,
    getGeoLocation,
    onAddressLineChange,
    onFocus
} from './service';

class AddressesField extends React.Component {
    addressRef = React.createRef();

    constructor(props) {
        super(props);
        this.displayName = 'AddressesField';

        this.getSuggest = throttle(getSuggest.bind(this), 300);
        this.setContext = setContext.bind(this);
        this.getOptions = getOptions.bind(this);
        this.clearContext = clearContext.bind(this);
        this.getSuggestValue = getSuggestValue.bind(this);
        this.deleteAddress = deleteAddress.bind(this);
        this.isDeliveryAddress = isDeliveryAddress.bind(this);
        this.updatedDeliveryAddress = updatedDeliveryAddress.bind(this);
        this.getGeoLocation = getGeoLocation.bind(this);
        this.onAddressLineChange = onAddressLineChange.bind(this);
        this.onFocus = onFocus.bind(this);
    }

    render() {
        const {
            address = {},
            context,
            id,
            isGeoLocationAvailable,
            geoLocationUpdateState,
            errors,
            isMobile
        } = this.props;
        const isDelivery = this.isDeliveryAddress();
        const options = this.getOptions();
        const placeholder = isDelivery
            ? i18n('Profile.addresses.placeholder.address.delivery')
            : i18n('Profile.addresses.placeholder.address.my');
        const showSuggest = Boolean(context && context === id && options.length);
        const isCanBeDeleted =
            address.id !== DEFAULT_DELIVERY_ADDRESS_ID && address.addressLine && !geoLocationUpdateState.progress;

        return (
            <div className='addresses-edit-control-wrap' data-t={`address:${isDelivery ? 'delivery' : id}`}>
                {!isDelivery && (
                    <div className='addresses-edit-control addresses-edit-control-type'>
                        {`${ADDRESS_TYPES[address.id] || ''}: `}
                    </div>
                )}
                <div
                    ref={this.addressRef}
                    className={classnames('addresses-edit-control addresses-edit-control-field', {
                        'addresses-edit-control-field__delivery': isDelivery
                    })}
                >
                    <Input
                        className={isGeoLocationAvailable ? 'addresses-edit-control__location' : ''}
                        placeholder={placeholder}
                        id='addressLine'
                        value={address.editedAddressLine}
                        name='addressLine'
                        controlRef={(field) => (this.addressLine = field)}
                        disabled={geoLocationUpdateState.progress}
                        autoComplete='off'
                        onChange={this.onAddressLineChange}
                        onBlur={this.clearContext}
                        onFocus={this.onFocus}
                        size='l'
                    />

                    {isGeoLocationAvailable && (
                        <span
                            className='addresses-edit-control-location'
                            role='button'
                            onClick={this.getGeoLocation}
                            aria-label={i18n('Profile.addresses.get-current-location.aria-label')}
                        />
                    )}

                    <Popup
                        className='addresses-edit-suggest'
                        visible={showSuggest}
                        target='anchor'
                        anchor={this.addressRef}
                        directions={isMobile ? ['bottom-left'] : ['bottom-left', 'top-left']}
                        zIndex={9999}
                        style={{maxWidth: '700px'}}
                    >
                        {showSuggest && (
                            <Menu
                                onChange={this.getSuggestValue}
                                size='s'
                                dataT='addresses-edit-suggest'
                                items={options}
                                className='addresses-suggest'
                            />
                        )}
                    </Popup>
                </div>
                {isDelivery && (
                    <div className='addresses-edit-control addresses-edit-control-field__short' data-t='address:flat'>
                        <Input
                            size='l'
                            autoComplete='off'
                            placeholder={i18n('Profile.addresses.placeholder.address.flat')}
                            id='flat'
                            value={address.editedAddressFlat}
                            name='flat'
                            disabled={geoLocationUpdateState.progress}
                            controlRef={(field) => (this.flat = field)}
                            onChange={this.updatedDeliveryAddress}
                        />
                    </div>
                )}

                {geoLocationUpdateState.id === id && geoLocationUpdateState.progress && (
                    <span
                        className={classnames('addresses-edit-control-location-spin', {
                            'addresses-edit-control-location-spin': isDelivery
                        })}
                    >
                        <Spin progress={true} size='s' />
                    </span>
                )}
                {isCanBeDeleted && (
                    <span
                        className={classnames('addresses-edit-control-delete', {
                            'addresses-edit-control-delete__delivery': isDelivery
                        })}
                        role='button'
                        aria-label={i18n('Profile.addresses.delete-location.aria-label')}
                        onClick={this.deleteAddress}
                    />
                )}
                {errors && (
                    <div className='p-control-error-block'>
                        {errors.map((code) => (
                            <div className='p-control-error-block' key={code}>
                                {ERRORS[code] || ERRORS.internal}
                            </div>
                        ))}
                    </div>
                )}
            </div>
        );
    }
}

AddressesField.propTypes = {
    address: PropTypes.shape({
        addressLine: PropTypes.string,
        flat: PropTypes.string
    }),
    dispatch: PropTypes.func.isRequired,
    suggest: PropTypes.array,
    context: PropTypes.string,
    id: PropTypes.string,
    isGeoLocationAvailable: PropTypes.bool,
    geoLocationUpdateState: PropTypes.object,
    errors: PropTypes.array,
    isMobile: PropTypes.bool
};

export default AddressesField;
