import React from 'react';
import PropTypes from 'prop-types';

import {Spin} from '@components/Spin';
import {Link as LegoLink} from '@components/Link';
import {Modal} from '@components/Modal';

import Link from '@blocks/morda/components/link/link.jsx';
import Address from './components/address';
import AddressesControl from './components/addresses_control';
import Map from '@blocks/morda/components/map/map.jsx';

import {EDIT_MODE_ADDRESSES, EDIT_MODE_DELIVERY_ADDRESSES, ADDRESSES_LINK} from './actions';

import {showAddresses, showDeliveryAddresses, setEditMode, sendMetrics} from './service';

const MAX_DELIVERY_ADDRESSES = 1;
const ZERO = 0;

export class Addresses extends React.Component {
    constructor(props) {
        super(props);

        this.showAddresses = showAddresses.bind(this);
        this.showDeliveryAddresses = showDeliveryAddresses.bind(this);
        this.setEditMode = setEditMode.bind(this);
        this.sendMetrics = sendMetrics.bind(this);
        this.getAddDefaultsLink = this.getAddDefaultsLink.bind(this);
        this.getAddDeliveryLink = this.getAddDeliveryLink.bind(this);
    }

    componentWillReceiveProps() {
        if (this.moreLink) {
            this.moreLink.forceUpdate();
        }
    }

    getAddDefaultsLink(isPhone, successConnect, tunePlaces) {
        if (isPhone && successConnect) {
            return (
                <Link onClick={this.showAddresses} url={ADDRESSES_LINK}>
                    {i18n('Profile.addresses.add.defaults')}
                </Link>
            );
        }

        if (successConnect) {
            return (
                <LegoLink pseudo={true} onClick={this.showAddresses}>
                    {i18n('Profile.addresses.add.defaults')}
                </LegoLink>
            );
        }

        return <LegoLink href={tunePlaces}>{i18n('Profile.addresses.add.defaults')}</LegoLink>;
    }

    getAddDeliveryLink(isPhone, successConnect, addressesLink, marketAddresses) {
        if (isPhone && successConnect) {
            return (
                <Link onClick={this.showDeliveryAddresses} url={addressesLink}>
                    {i18n('Profile.addresses.add.delivery')}
                </Link>
            );
        }

        if (successConnect) {
            return (
                <LegoLink pseudo={true} onClick={this.showDeliveryAddresses}>
                    {i18n('Profile.addresses.add.delivery')}
                </LegoLink>
            );
        }

        return <LegoLink href={marketAddresses}>{i18n('Profile.addresses.add.delivery')}</LegoLink>;
    }

    render() {
        const {addresses = {}, dispatch, edit, settings = {}, map, common} = this.props;
        const {tld, ua = {}, tunePlaces = '', marketAddresses = ''} = settings;
        const {isMobile, isTouch, isTablet} = ua;
        const isPhone = (isMobile || isTouch) && !isTablet;
        const {
            delivery = [],
            home = {
                id: 'home'
            },
            work = {
                id: 'work'
            },
            progress,
            successConnect = false
        } = addresses;
        const deliveryAddressesTail = delivery.length - MAX_DELIVERY_ADDRESSES;
        const deliveryAddressesLinkText = i18n(
            {
                count: deliveryAddressesTail,
                one: 'Profile.addresses.more.one',
                some: 'Profile.addresses.more.some',
                many: 'Profile.addresses.more.many',
                none: ''
            },
            deliveryAddressesTail
        );
        const TUNE_PLACES = tunePlaces.replace('%tld%', tld);
        const MARKET_ADDRESSES = marketAddresses.replace('%tld%', tld);
        const mapAddresses = [];

        if (home.longitude) {
            mapAddresses.push(home);
        }

        if (work.longitude) {
            mapAddresses.push(work);
        }

        return (
            <div className='section addresses'>
                <span className='section-icon' />

                <div className='section-head'>
                    <h2 className='section-title'>{i18n('Profile.addresses.title.short')}</h2>
                </div>

                <div className='addresses-body' data-t='address-widget:work-home'>
                    {!home.longitude && !work.longitude ? (
                        <div>
                            <div className='p-control-link p-control-link__bold'>
                                <span
                                    onMouseDown={() => this.sendMetrics('add_navi')}
                                    data-t='address-widget:add-home-work'
                                >
                                    {this.getAddDefaultsLink(isPhone, successConnect, TUNE_PLACES)}
                                </span>
                            </div>
                            <div className='section-body-info section-body-info__bottom_shift'>
                                {i18n('Profile.addresses.info.defaults')}
                            </div>
                        </div>
                    ) : (
                        <div>
                            <Map
                                className='addresses-map'
                                points={mapAddresses}
                                settings={settings}
                                width={260}
                                height={76}
                            />
                            <Address dispatch={dispatch} address={home} isPhone={isPhone} addIfEmpty={true} />
                            <Address dispatch={dispatch} address={work} isPhone={isPhone} addIfEmpty={true} />
                            <div className='p-control-link'>
                                <span onMouseDown={() => this.sendMetrics('edit_navi')} data-t='address-widget:edit'>
                                    {isPhone ? (
                                        <Link onClick={this.showAddresses} url={ADDRESSES_LINK}>
                                            {i18n('Profile.addresses.edit')}
                                        </Link>
                                    ) : (
                                        <LegoLink pseudo={true} onClick={this.showAddresses}>
                                            {i18n('Profile.addresses.edit')}
                                        </LegoLink>
                                    )}
                                </span>
                            </div>
                        </div>
                    )}
                </div>

                {Boolean(tld === 'ru') &&
                    (!delivery.length ? (
                        <div className='addresses-body' data-t='address-widget:delivery'>
                            <div className='p-control-link'>
                                <span
                                    onMouseDown={() => this.sendMetrics('add_delivery')}
                                    data-t='address-widget:add-delivery'
                                >
                                    {this.getAddDeliveryLink(isPhone, successConnect, ADDRESSES_LINK, MARKET_ADDRESSES)}
                                </span>
                            </div>
                            <div className='section-body-info section-body-info__bottom_shift'>
                                {i18n('Profile.addresses.info.delivery')}
                            </div>
                        </div>
                    ) : (
                        <div>
                            <Address address={delivery[ZERO]} />

                            {Boolean(deliveryAddressesTail > ZERO) && (
                                <div className='p-control-link'>
                                    {isPhone ? (
                                        <Link onClick={this.showDeliveryAddresses} url={ADDRESSES_LINK}>
                                            {deliveryAddressesLinkText}
                                        </Link>
                                    ) : (
                                        <LegoLink
                                            ref={(moreLink) => (this.moreLink = moreLink)}
                                            onClick={this.showDeliveryAddresses}
                                        >
                                            {deliveryAddressesLinkText}
                                        </LegoLink>
                                    )}
                                </div>
                            )}

                            <div className='p-control-link addresses-add'>
                                <span onMouseDown={() => this.sendMetrics('edit_delivery')}>
                                    {isPhone ? (
                                        <Link onClick={this.showDeliveryAddresses} url={ADDRESSES_LINK}>
                                            {i18n('Profile.addresses.edit')}
                                        </Link>
                                    ) : (
                                        <LegoLink onClick={this.showDeliveryAddresses}>
                                            {i18n('Profile.addresses.edit')}
                                        </LegoLink>
                                    )}
                                </span>
                            </div>
                        </div>
                    ))}

                <Spin progress={progress} size='s' />

                <Modal
                    visible={edit === EDIT_MODE_ADDRESSES || edit === EDIT_MODE_DELIVERY_ADDRESSES}
                    zIndexGroupLevel={-0.5}
                    dataT='addresses'
                >
                    <AddressesControl
                        modal={true}
                        map={map}
                        settings={settings}
                        addresses={addresses}
                        dispatch={dispatch}
                        common={common}
                        edit={edit}
                    />
                </Modal>
            </div>
        );
    }
}

Addresses.propTypes = {
    settings: PropTypes.shape({
        ua: PropTypes.shape({
            isMobile: PropTypes.bool,
            isTouch: PropTypes.bool,
            isTablet: PropTypes.bool
        })
    }),
    dispatch: PropTypes.func.isRequired,
    edit: PropTypes.string,
    map: PropTypes.object,
    addresses: PropTypes.object,
    common: PropTypes.object
};
