import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {setEditMode} from '@blocks/common/actions';
import {push} from 'react-router-redux';
import utils, {tldMapping} from '@blocks/utils';
import classnames from 'classnames';
import {
    getTokensList,
    revokeToken,
    disableApplicationPasswords,
    activateApplicationPasswords,
    countAppPasswords
} from './index';
import {updateTokenTab, showDisableScreen} from './actions';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';
import {LINK_CLASS_NAME} from '@blocks/morda/components/link/link.jsx';

export class AppPasswordsList extends Component {
    constructor(props) {
        super(props);
        this.showDisableScreen = this.showDisableScreen.bind(this);
        this.createNewAppPassword = this.createNewAppPassword.bind(this);
        this.redirectTo = this.redirectTo.bind(this);
    }

    componentDidMount() {
        const {dispatch, settings} = this.props;
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;
        const isStandAlonePage = this.props.common && this.props.common.currentPage !== '/profile';

        // invokes for single page /profile/apppasswords-list
        if (isPhone || isStandAlonePage) {
            dispatch(countAppPasswords());
            dispatch(getTokensList());
        } else {
            dispatch(showDisableScreen(false));
        }
    }

    createNewAppPassword() {
        const {settings, dispatch} = this.props;
        const isStandAlonePage = this.props.common && this.props.common.currentPage !== '/profile';
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;

        if (isPhone || isStandAlonePage) {
            if (typeof window !== 'undefined') {
                // using push('/profile') causes a bug when a random password type is chosen
                this.redirectTo('/profile/generate-apppassword');
            }
            return;
        }
        dispatch(setEditMode('apppwd-create'));
    }

    redirectTo(url) {
        const retpath = this.props.common && this.props.common.retpath;

        let redirectUrl = url;

        if (retpath) {
            redirectUrl += `?retpath=${encodeURIComponent(retpath)}`;
        }

        window.location.href = redirectUrl;
    }

    showDisableScreen() {
        this.props.dispatch(showDisableScreen(true));
    }

    render() {
        const {closeModal, dispatch, appPasswords, settings = {}} = this.props;
        const {ua = {}, tld} = settings;
        const {isMobile} = ua;
        const isPhone = (isMobile || ua.isTouch) && !ua.isTablet;
        const isEnabled = this.props.isEnabled || (this.props.access && this.props.access.isAppPasswordsEnabled);
        const is2faEnabled = this.props.is2faEnabled || (this.props.access && this.props.access.is2faEnabled);
        const is2faDisabled = !is2faEnabled;
        const count = appPasswords && appPasswords.tokens.appPasswordsCount;
        const showDisable = appPasswords.tokens.showDisableScreen;
        const tokensList = appPasswords.tokens.list;
        const showFullList = count > 0 && !showDisable;
        const showEmpty = isEnabled && count === 0 && !showDisable;
        const isStandAlonePage = this.props.common && this.props.common.currentPage !== '/profile';

        return (
            <div className={classnames('p-control-form manage-app-passwords')}>
                {(() => {
                    if (showEmpty) {
                        return (
                            <AppPasswordsEmptyList
                                createNew={this.createNewAppPassword}
                                disable={this.showDisableScreen}
                                tld={tld}
                                is2faDisabled={is2faDisabled}
                                dispatch={dispatch}
                                isMobile={isMobile}
                            />
                        );
                    }

                    if (showFullList) {
                        return (
                            <AppPasswordsFullList
                                tokensList={tokensList}
                                tld={tld}
                                createNew={this.createNewAppPassword}
                                is2faDisabled={is2faDisabled}
                                count={count}
                                disable={this.showDisableScreen}
                                isStandAlonePage={isStandAlonePage}
                                dispatch={dispatch}
                                isMobile={isMobile}
                            />
                        );
                    }

                    if (showDisable) {
                        return (
                            <AppPasswordsDisable
                                isPhone={isPhone}
                                cancel={closeModal}
                                isStandAlonePage={isStandAlonePage}
                                dispatch={dispatch}
                                isMobile={isMobile}
                            />
                        );
                    }

                    return <AppPasswordsDefault dispatch={dispatch} isMobile={isMobile} />;
                })()}
            </div>
        );
    }
}

class AppPasswordsDefault extends Component {
    constructor(props) {
        super(props);
        this.activateAppPasswords = this.activateAppPasswords.bind(this);
    }

    activateAppPasswords() {
        this.props.dispatch(activateApplicationPasswords());
    }

    render() {
        const {isMobile} = this.props;

        return (
            <div>
                <h2 className='section-title'>{i18n('Profile.apppasswords.enable')}</h2>
                <div className='section-info'>{i18n('Profile.access.widget-default-description')}</div>

                <div className='app-passwords-list_default'>
                    <div className='app-passwords-list_content align-center'>
                        <Button
                            text={i18n('Profile.apppasswords.enable')}
                            size='l'
                            view='action'
                            type='button'
                            onClick={this.activateAppPasswords}
                            width={isMobile ? 'max' : ''}
                        />
                    </div>
                </div>
            </div>
        );
    }
}

class AppPasswordsEmptyList extends Component {
    render() {
        const {tld, isMobile} = this.props;
        const supportTld = tldMapping[tld] || 'ru';
        const helpLink = i18n('Profile.access.widget-main-description')
            .replace('%s', `https://yandex.${supportTld}/support/passport/authorization/app-passwords.html`)
            .replace('%1', LINK_CLASS_NAME);
        const aboutText = i18n('Profile.access.widget-empty-description');

        return (
            <div>
                <h2 className='section-title'>{i18n('Profile.access.appwd-title')}</h2>
                <div className='section-info' dangerouslySetInnerHTML={{__html: helpLink}} />
                <div className='content' dangerouslySetInnerHTML={{__html: aboutText}} />
                <div className='app-passwords-list_empty'>
                    <div className='controls-wrapper'>
                        <div className='right-aligned'>
                            <Button
                                text={i18n('Profile.access.widget-create_new-btn')}
                                size='l'
                                view='action'
                                type='button'
                                width={isMobile ? 'max' : 'auto'}
                                onClick={this.props.createNew}
                            />
                        </div>
                        {Boolean(this.props.is2faDisabled) && (
                            <div className='left-aligned'>
                                <Button
                                    text={i18n('Profile.apppasswords.disable')}
                                    size='l'
                                    view='pseudo'
                                    type='button'
                                    width={isMobile ? 'max' : 'auto'}
                                    onClick={this.props.disable}
                                />
                            </div>
                        )}
                        <div className='p-clear' />
                    </div>
                </div>
            </div>
        );
    }
}

class AppPasswordsFullList extends Component {
    constructor(props) {
        super(props);
        this.getDate = this.getDate.bind(this);
        this.showInfo = this.showInfo.bind(this);
        this.revokeToken = this.revokeToken.bind(this);
        this.preventFold = this.preventFold.bind(this);
    }

    getDate(timestamp) {
        if (!timestamp) {
            return '';
        }

        return utils
            .dateToText(new Date(timestamp * 1000), {
                year: true,
                time: true
            })
            .toLowerCase();
    }

    showInfo(e) {
        const {dispatch} = this.props;
        const id = e.currentTarget.getAttribute('data-tokenInfo');

        dispatch(updateTokenTab(id));
    }

    preventFold(e) {
        e.stopPropagation();
    }

    revokeToken(e) {
        const tokenId = e.currentTarget.getAttribute('data-tokenId');

        this.props.dispatch(revokeToken(tokenId));
    }

    render() {
        const {tld, isMobile, tokensList, count, isStandAlonePage} = this.props;
        const supportTld = tldMapping[tld] || 'ru';

        const helpLink = i18n('Profile.access.widget-main-description')
            .replace('%s', `https://yandex.${supportTld}/support/passport/authorization/app-passwords.html`)
            .replace('%1', LINK_CLASS_NAME);

        return (
            <div>
                <h2 className='section-title'>{i18n('Profile.access.appwd-title')}</h2>
                <div className='section-info' dangerouslySetInnerHTML={{__html: helpLink}} />
                <div className='app-passwords-list_items'>
                    <div className='app-passwords-list_content'>
                        <span className='personal-spinner'>
                            <Spin size='s' progress={tokensList.length === 0} />
                        </span>
                        <div className='info-list'>
                            <div className='info-list_head'>
                                <span className='title app-password-type'>
                                    {i18n('Profile.access.widget-list-title')}
                                </span>
                                <span className='title app-password-protocol'>
                                    {i18n('Profile.access.widget-list-protocol')}
                                </span>
                                <span className='title app-password-created-date'>
                                    {i18n('Profile.access.widget-list-created-date')}
                                </span>
                            </div>
                            <div
                                className={classnames('info-list-wrapper', {
                                    add_scroll: count > 3 && !isStandAlonePage
                                })}
                            >
                                {Boolean(tokensList) &&
                                    tokensList.map((token = {}) => (
                                        <div
                                            className={classnames('info-list_item', {active: token.tabOpened})}
                                            key={token.tokenId}
                                            ref={`token-${token.tokenId}`}
                                            data-tokenInfo={token.tokenId}
                                            onClick={this.showInfo}
                                        >
                                            <span className='toggler' />
                                            <div className='token-name'>
                                                {token.name && <strong>{token.name} </strong>}
                                                {Boolean(token.deviceName) && (
                                                    <span>
                                                        — <span className='device-name'>{token.deviceName}</span>
                                                    </span>
                                                )}
                                            </div>
                                            <div className='protokol'>
                                                {token.scopes && token.scopes.map((scope) => scope.scope).join(', ')}
                                            </div>
                                            <div className='created-date'>{this.getDate(token.createTime)}</div>
                                            <div className='extend-info' onClick={this.preventFold}>
                                                <p>
                                                    {i18n('Profile.access.widget-tokenId')} {token.deviceId}
                                                </p>
                                                <span
                                                    className={LINK_CLASS_NAME}
                                                    data-tokenId={token.tokenId}
                                                    onClick={this.revokeToken}
                                                >
                                                    {i18n('Profile.access.widget-delete-pass-btn')}
                                                </span>
                                            </div>
                                        </div>
                                    ))}
                            </div>
                        </div>
                        <div className='controls-wrapper'>
                            {Boolean(this.props.is2faDisabled) && (
                                <div className='left-aligned'>
                                    <Button
                                        text={i18n('Profile.apppasswords.disable')}
                                        size='l'
                                        view='pseudo'
                                        type='button'
                                        width={isMobile ? 'max' : 'auto'}
                                        onClick={this.props.disable}
                                    />
                                </div>
                            )}

                            <div className='right-aligned'>
                                <Button
                                    text={i18n('Profile.access.widget-create_new-btn')}
                                    size='l'
                                    view='action'
                                    type='button'
                                    width={isMobile ? 'max' : 'auto'}
                                    onClick={this.props.createNew}
                                />
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        );
    }
}

class AppPasswordsDisable extends Component {
    constructor(props) {
        super(props);
        this.disableAppPasswords = this.disableAppPasswords.bind(this);
    }

    disableAppPasswords() {
        const {isPhone, dispatch, isStandAlonePage} = this.props;

        dispatch(disableApplicationPasswords());
        if (isPhone || isStandAlonePage) {
            dispatch(push('/profile'));
        }
    }

    render() {
        const {isMobile} = this.props;
        const disableText = i18n('Profile.access.widget-disable-text');

        return (
            <div>
                <h2 className='section-title'>{i18n('Profile.access.widget-disable-title')}</h2>
                <div className='section-info' dangerouslySetInnerHTML={{__html: disableText}} />
                <div className='app-passwords-list_disable'>
                    <div className='content align-center'>
                        <div className='controls-wrapper'>
                            <div className='left-aligned'>
                                <Button
                                    text={i18n('_AUTH_.common.cancel')}
                                    size='l'
                                    view='pseudo'
                                    type='button'
                                    width={isMobile ? 'max' : 'auto'}
                                    onClick={this.props.cancel}
                                />
                            </div>
                            <div className='right-aligned'>
                                <Button
                                    text={i18n('Profile.apppasswords.disable')}
                                    size='l'
                                    view='action'
                                    type='button'
                                    width={isMobile ? 'max' : 'auto'}
                                    onClick={this.disableAppPasswords}
                                />
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        );
    }
}

AppPasswordsList.propTypes = {
    closeModal: PropTypes.func.isRequired,
    isEnabled: PropTypes.bool,
    is2faEnabled: PropTypes.bool,
    edit: PropTypes.string,
    dispatch: PropTypes.func.isRequired,
    settings: PropTypes.object.isRequired,
    appPasswords: PropTypes.shape({
        createAppPassword: PropTypes.object.isRequired,
        tokens: PropTypes.shape({
            appPasswordsCount: PropTypes.number,
            list: PropTypes.array,
            showDisableScreen: PropTypes.bool
        })
    }),
    common: PropTypes.object,
    access: PropTypes.object
};

AppPasswordsFullList.propTypes = {
    dispatch: PropTypes.func.isRequired,
    createNew: PropTypes.func.isRequired,
    disable: PropTypes.func.isRequired,
    tld: PropTypes.string,
    tokensList: PropTypes.array.isRequired,
    is2faDisabled: PropTypes.bool,
    isStandAlonePage: PropTypes.bool,
    isMobile: PropTypes.bool,
    count: PropTypes.number
};

AppPasswordsDefault.propTypes = {
    dispatch: PropTypes.func.isRequired,
    isMobile: PropTypes.bool
};

AppPasswordsDisable.propTypes = {
    isPhone: PropTypes.bool,
    cancel: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    isStandAlonePage: PropTypes.bool,
    isMobile: PropTypes.bool
};

AppPasswordsEmptyList.propTypes = {
    dispatch: PropTypes.func.isRequired,
    createNew: PropTypes.func.isRequired,
    disable: PropTypes.func.isRequired,
    is2faDisabled: PropTypes.bool,
    tld: PropTypes.string,
    isMobile: PropTypes.bool
};
