import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import {Input} from '@components/Input';
import {ErrorMessage} from './error.jsx';
import sendMetrics from '../helpers/sendAvatarMetrics';
import {avatarCropper} from '../helpers/avatarCropper';
import {setAvatarError} from '../actions/index';

export class AddUrl extends PureComponent {
    constructor(props) {
        super(props);

        this.checkError = this.checkError.bind(this);
        this.handleUrl = this.handleUrl.bind(this);
        this.handleValue = this.handleValue.bind(this);
        this.handleEvent = this.handleEvent.bind(this);

        this.state = {
            text: ''
        };
    }

    componentWillReceiveProps(nextProps) {
        if (nextProps.hasAvatarChanged) {
            this.setState(() => ({
                text: ''
            }));
        }
    }

    handleValue(event) {
        const {target = {}} = event;
        const {value} = target;

        if (!value) {
            this.props.dispatch(avatarCropper.destroyCropImage());
        }

        this.setState(() => ({
            text: value
        }));
        this.props.dispatch(setAvatarError(''));
        this.props.checkField(value);
    }

    handleEvent(event) {
        if (event.keyCode === 13) {
            this.handleUrl();
        }
    }

    checkError(error) {
        const errors = [
            'error-url.invalid',
            'error-url.file.size',
            'error-url.empty',
            'error-url.image_size',
            'error-limit'
        ];

        return errors.includes(error);
    }

    handleUrl() {
        this.props.handleUrl(this.state.text);
    }

    render() {
        const urlPlaceholderText = 'https://example.com/img.jpg';
        const {errorType, hasAvatarChanged, disabled, isLoading, isProcessing, isByUrl} = this.props;
        const errorIsVisible = this.checkError(errorType);
        const fieldText = hasAvatarChanged ? '' : this.state.text;

        return (
            <div className='avatar-load_url'>
                <div className='field-text'>{i18n('Profile.avatar.set-url-text')}</div>
                <div
                    className={classnames('edit-avatar_url-wrapper', {input_filled: this.state.text})}
                    onKeyUp={this.handleEvent}
                >
                    <Input
                        size='l'
                        id='url'
                        value={fieldText}
                        name='url'
                        type='url'
                        disabled={disabled || (isProcessing && !isByUrl)}
                        hasClear={true}
                        placeholder={urlPlaceholderText}
                        onChange={this.handleValue}
                        iconRight={
                            <span onMouseDown={sendMetrics} id='profile_page_load_avatar-url' data-msg='Загрузить'>
                                <button
                                    type='button'
                                    className='edit-avatar_upload-url'
                                    disabled={isLoading || (isProcessing && !isByUrl)}
                                    onClick={this.handleUrl}
                                    title={i18n('Profile.avatar.choose-file')}
                                />
                            </span>
                        }
                    />
                </div>
                <ErrorMessage visible={errorIsVisible} errorType={errorType} />
            </div>
        );
    }
}

AddUrl.propTypes = {
    isLoading: PropTypes.bool.isRequired,
    isProcessing: PropTypes.bool,
    isByUrl: PropTypes.bool.isRequired,
    disabled: PropTypes.bool,
    handleUrl: PropTypes.func.isRequired,
    errorType: PropTypes.string,
    hasAvatarChanged: PropTypes.bool,
    dispatch: PropTypes.func.isRequired,
    checkField: PropTypes.func.isRequired
};

export default connect(({changeAvatar: {status}}) => ({
    isLoading: status === 'loading',
    isProcessing: status === 'processing'
}))(AddUrl);
