import React, {Component} from 'react';
import PropTypes from 'prop-types';
import api from '@blocks/api';
import {Button} from '@components/Button';
import classnames from 'classnames';
import {setEditMode, goToRetpath} from '../../../common/actions';
import getInitialAvatarsTrack from '../actions/getInitialAvatarsTrack';
import sendMetrics from '../helpers/sendAvatarMetrics';
import avatarByUrl from '../actions/loadAvatarByUrl';
import EditAvatar from './form.jsx';
import {updateAvatarStatus} from '../actions';
import {avatarCropper} from '../helpers/avatarCropper';
import checkRetpath from '../actions/checkRetpath';

export class AvatarInfo extends Component {
    componentDidMount() {
        this.props.dispatch(getInitialAvatarsTrack());
    }

    render() {
        const {settings, dispatch, changeAvatar, edit, modal} = this.props;
        const isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;
        const isStandAlonePage = settings && settings.location !== '/profile/';
        const isCropNotSupported = changeAvatar.isCropNotSupported;
        const retpath = this.props.retpath || (this.props.common && this.props.common.retpath);

        return (
            <div className='avatar-wrap'>
                <ChangeAvatar
                    modal={modal}
                    edit={edit}
                    changeAvatar={changeAvatar}
                    isPhone={isPhone}
                    isCropNotSupported={isCropNotSupported}
                    isStandAlonePage={isStandAlonePage}
                    retpath={retpath}
                    dispatch={dispatch}
                />
            </div>
        );
    }
}

export class ChangeAvatar extends Component {
    constructor(props) {
        super(props);
        this.closeModal = this.closeModal.bind(this);
        this.handleSaveAction = this.handleSaveAction.bind(this);
    }

    closeModal() {
        const {dispatch, isPhone, isStandAlonePage} = this.props;

        if (isPhone || isStandAlonePage) {
            dispatch(goToRetpath());
        }
        dispatch(setEditMode(null));
        dispatch(avatarCropper.destroyCropImage());
    }

    componentDidUpdate(prevProps) {
        const {dispatch, isPhone, changeAvatar} = this.props;
        const hasChanged = changeAvatar.hasChanged !== prevProps.changeAvatar.hasChanged;

        if (hasChanged && isPhone) {
            dispatch(goToRetpath());
        }
    }

    handleSaveAction() {
        const {dispatch, changeAvatar} = this.props;
        const {status, loadUrlPath, url, track_id} = changeAvatar;

        if (status === 'deleted' && !changeAvatar.error) {
            this.closeModal();
        }
        if (status !== 'processing' && loadUrlPath) {
            dispatch(avatarByUrl());
            return;
        }

        if (url && status === 'processing') {
            dispatch(avatarCropper.cropAvatar());
            dispatch(updateAvatarStatus('loading'));
            api.writeStatbox({
                mode: 'profile_change_avatar',
                track_id,
                action: 'update'
            });
            return;
        }

        dispatch(checkRetpath());
    }

    render() {
        const {
            modal,
            changeAvatar,
            edit,
            isStandAlonePage,
            isCropNotSupported,
            retpath,
            dispatch,
            isPhone = false
        } = this.props;
        const blockClass = classnames({
            'p-control-form': true,
            'p-control-modal avatar-modal': modal
        });
        const isLoading = changeAvatar.status === 'loading';
        const isProcessing = changeAvatar.status === 'processing';
        const showButtons = !isPhone || (isPhone && isProcessing) || (isPhone && isLoading);
        const cancelBtnTxt =
            isStandAlonePage && retpath ? i18n('Profile.common.return-to-service') : i18n('_AUTH_.common.cancel');

        return (
            <div className={blockClass}>
                <h2 className='section-title avatar-modal__title'>{i18n('Profile.avatars.title')}</h2>
                <EditAvatar
                    avatarInfo={changeAvatar}
                    edit={edit}
                    isPhone={isPhone}
                    isCropNotSupported={isCropNotSupported}
                    dispatch={dispatch}
                />
                {showButtons && !isCropNotSupported && (
                    <div className='avatar-buttons'>
                        <div className='edit-avatar_btn-save'>
                            <span id='profile_page_save_avatar' data-msg='Сохранить' onMouseDown={sendMetrics}>
                                <Button
                                    view='action'
                                    size='l'
                                    width={isPhone ? 'max' : 'auto'}
                                    disabled={isLoading && isPhone}
                                    onClick={this.handleSaveAction}
                                >
                                    {i18n('_AUTH_.common.save')}
                                </Button>
                            </span>
                        </div>
                        <div className='edit-avatar_btn-cancel'>
                            <Button
                                view='pseudo'
                                size='l'
                                width={isPhone ? 'max' : 'auto'}
                                disabled={isLoading && isPhone}
                                onClick={this.closeModal}
                            >
                                {cancelBtnTxt}
                            </Button>
                        </div>
                    </div>
                )}
                {isPhone && <div className={classnames('avatars__paranja', {avatars__paranja_visible: isLoading})} />}
            </div>
        );
    }
}

ChangeAvatar.propTypes = {
    changeAvatar: PropTypes.shape({
        backupAvatar: PropTypes.object,
        defaultUrl: PropTypes.string,
        error: PropTypes.string,
        hasChanged: PropTypes.bool,
        id: PropTypes.number,
        isByUrl: PropTypes.bool,
        status: PropTypes.string,
        track_id: PropTypes.string,
        url: PropTypes.string,
        retpath: PropTypes.string,
        isCropNotSupported: PropTypes.bool,
        loadUrlPath: PropTypes.string
    }),
    isCropNotSupported: PropTypes.bool,
    dispatch: PropTypes.func,
    edit: PropTypes.string,
    modal: PropTypes.bool,
    isPhone: PropTypes.bool,
    isStandAlonePage: PropTypes.bool,
    retpath: PropTypes.string
};

AvatarInfo.propTypes = {
    changeAvatar: PropTypes.shape({
        backupAvatar: PropTypes.object,
        common: PropTypes.object,
        defaultUrl: PropTypes.string,
        error: PropTypes.string,
        hasChanged: PropTypes.bool,
        id: PropTypes.number,
        isByUrl: PropTypes.bool,
        status: PropTypes.string,
        track_id: PropTypes.string,
        url: PropTypes.string,
        retpath: PropTypes.string,
        queryUid: PropTypes.string,
        isCropNotSupported: PropTypes.bool,
        loadUrlPath: PropTypes.string
    }),
    dispatch: PropTypes.func,
    edit: PropTypes.string,
    modal: PropTypes.bool,
    common: PropTypes.object,
    settings: PropTypes.object,
    uid: PropTypes.number,
    retpath: PropTypes.string
};
